//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  can_device_config (
    input   wire    reset_n,
    input   wire    ck,
    input   wire    carry_1ms,

    input   wire    start,
    output  logic   done,

    spi_cont_if.out cont_out
);

`include "MCP2515_types.svh"
`include "MCP2515.svh"

`define  BIT_RATE_1M
`include "can_bit_rate.svh"

localparam  WAIT_AFTER_RESET_MS = 5'd20; // reset 後の wait (5ms)

localparam  SREG_BYTE_LEN = 6;
localparam  PAD0_FULL = {(SREG_BYTE_LEN){8'b0}};
localparam  PAD0_LES1 = {(SREG_BYTE_LEN-1){8'b0}};
localparam  PAD0_LES4 = {(SREG_BYTE_LEN-4){8'b0}};

typedef enum logic [3:0] {
    s_init,
    s_cmd_seq_start_wait,
    s_spi_port_request,
    s_spi_port_permit_wait,
    s_device_reset,
    s_wait_timer_start,
    s_wait_after_device_reset,
    s_config_1,
    s_config_2,
    s_can_com_start,
    s_spi_port_release,
    s_spi_port_release_wait,
    s_cmd_seq_done,

    s_spi_tx_proc,
    s_spi_tx_done_wait
} state_t;

//-----------------------------------------------------------
state_t curr_state, next_state, return_state;

logic   [(SREG_BYTE_LEN*8-1):0] spi_rx_data;
logic   [(SREG_BYTE_LEN*8-1):0] spi_tx_command;

logic   [7:0] total_byte_num;

logic   spi_tx_start;
logic   spi_tx_done;

logic   wait_timer_start;
logic   wait_time_up;

CNF1_t      CNF1;
CNF2_t      CNF2;
CNF3_t      CNF3;
CANINTE_t   CANINTE;
RXB0CTRL_t  RXB0CTRL;
CANCTRL_t   CANCTRL;

//-----------------------------------------------------------
shift_reg  #(
    .BYTE_LEN    (SREG_BYTE_LEN)
) 
spi_data_reg (
    .* ,
    .data_load      (spi_tx_start),
    .data_in        (spi_tx_command),
    .data_out       (spi_rx_data),
    .shift_carry    (cont_out.shift_carry),
    .shift_in       (cont_out.miso),
    .shift_out      (cont_out.mosi)
);
assign  cont_out.byte_num = total_byte_num;
assign  cont_out.start    = spi_tx_start;
assign  spi_tx_done = cont_out.done; 


on_timer #(
    .CNT_BIT_WIDTH  ($bits(WAIT_AFTER_RESET_MS))
) 
wait_after_reset_timer (
    .* ,
    .carry  (carry_1ms),
    .delay  (WAIT_AFTER_RESET_MS),
    .in     (wait_timer_start),
    .out    (wait_time_up)
);



assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        next_state <= s_init; return_state <= s_init;
        cont_out.request <= 1'b0;
        total_byte_num <= 8'd0;
        spi_tx_start <= 1'b0;
        spi_tx_command <= PAD0_FULL;
        wait_timer_start <= 1'b0;
        CNF3 <= 8'b0; CNF2 <= 8'b0; CNF1 <= 8'b0; 
        CANINTE <= 8'b0; RXB0CTRL <= 8'b0; CANCTRL <= 8'b0;
    end
    else begin
        case ( curr_state )
            s_init: begin
                cont_out.request <= 1'b0;
                total_byte_num <= 8'd0;
                spi_tx_start <= 1'b0;
                wait_timer_start <= 1'b0;
                CNF3 <= 8'b0; CNF2 <= 8'b0; CNF1 <= 8'b0; 
                CANINTE <= 8'b0; RXB0CTRL <= 8'b0; CANCTRL <= 8'b0;
                next_state <= s_cmd_seq_start_wait;
                return_state <= s_init;
            end

            // start 条件を待つ。
            s_cmd_seq_start_wait: begin
                if ( start ) begin
                    next_state <= s_spi_port_request;
                end
            end

            // spi port の制御権を要求する。
            s_spi_port_request: begin
                cont_out.request <= 1'b1;
                next_state <= s_spi_port_permit_wait;
            end

            // spi port の制御権取得を待つ。
            s_spi_port_permit_wait: begin
                if ( cont_out.permit ) begin
                    next_state <= s_device_reset;
                end
            end

            s_device_reset: begin
                total_byte_num <= 8'd1;
                spi_tx_command <= {
                    CMD_RESET, 
                    PAD0_LES1
                };
                next_state <= s_spi_tx_proc;
                return_state <= s_wait_timer_start;
            end

            s_wait_timer_start: begin
                wait_timer_start <= 1'b1;
                next_state <= s_wait_after_device_reset;
            end

            s_wait_after_device_reset: begin
                if ( wait_time_up ) begin
                    wait_timer_start <= 1'b0;

                    CNF3 <= LP_CNF3;
                    CNF2 <= LP_CNF2;
                    CNF1 <= LP_CNF1;

                    CANINTE.TX2IE <= 1'b1; CANINTE.TX1IE <= 1'b1; CANINTE.TX0IE <= 1'b1;
                    CANINTE.RX1IE <= 1'b1; CANINTE.RX0IE <= 1'b1;

                    next_state <= s_config_1;
                end
            end

            // デバイスの基本設定
            s_config_1: begin
                total_byte_num <= 8'd6;
                spi_tx_command <= {
                    CMD_WRITE,
                    ADRS_CNF3,
                    CNF3,
                    CNF2,
                    CNF1,
                    CANINTE
                };

                RXB0CTRL.BUKT <= 1'b1;  // RXB0 が full なら RXB1 へ書き込む。
                next_state <= s_spi_tx_proc;
                return_state <= s_config_2;
            end

            // RX Buffer 0 の構成
            s_config_2: begin
                total_byte_num <= 8'd4;
                spi_tx_command <= {
                    CMD_BIT_MODIFY,
                    ADRS_RXB0CTRL,
                    RXB0CTRL,   // as mask
                    RXB0CTRL,   // as data
                    PAD0_LES4
                };

                CANCTRL.REQOP <= REQOP_NORMAL;
                next_state <= s_spi_tx_proc;
                return_state <= s_can_com_start;
            end

            // 通常モード(CAN 通信開始)への移行を要求する。
            s_can_com_start: begin
                total_byte_num <= 8'd4;
                spi_tx_command <= {
                    CMD_BIT_MODIFY,
                    ADRS_CANCTRL,
                    MASK_CAN_CTRL_REQOP,    // as mask
                    CANCTRL,                // as data
                    PAD0_LES4
                };
                next_state <= s_spi_tx_proc;
                return_state <= s_spi_port_release;
            end

            //---------------------------------------------------------
            // spi port の制御権を解放する。
            s_spi_port_release: begin
                cont_out.request <= 1'b0;
                next_state <= s_spi_port_release_wait;
            end

            // spi port の制御権解放を待つ。
            s_spi_port_release_wait: begin
                if ( ~cont_out.permit ) begin
                    next_state <= s_cmd_seq_done;
                end
            end

            // コマンド制御の終了(done)を出力する。
            s_cmd_seq_done: begin
                next_state <= s_init;
            end



            //----------------------------------------------------
            // spi 通信を行う処理。 終了後に return する。
            s_spi_tx_proc: begin
                // spi へ trig をかける
                spi_tx_start <= 1'b1;
                next_state <= s_spi_tx_done_wait; 
            end
            // spi制御の終了(spi done)を待つ。 
            s_spi_tx_done_wait: begin
                spi_tx_start <= 1'b0;
                if ( spi_tx_done ) begin
                    next_state <= return_state;
                end
            end
            //----------------------------------------------------


            default: begin
                next_state <= s_init;
            end

        endcase
    end
end


always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        done <= 1'b0; 
    end
    else begin
        if (curr_state == s_cmd_seq_done)
            done <= 1'b1;
        else
            done <= 1'b0; 
    end
end



endmodule

