//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

`include "can_device_status.svh"

module  can_device_status_read (
    input   wire    reset_n,
    input   wire    ck,
    input   wire    start,
    output  logic   done,
    output  can_dev_sts_t   can_dev_sts,
    output  logic   [2:0]   txbuf_busy,
    spi_cont_if.out         cont_out
);

//-----------------------------------------------------------
`include "MCP2515.svh"

localparam  SREG_BYTE_LEN = 4;
localparam  PAD0_FULL = {(SREG_BYTE_LEN){8'b0}};
localparam  PAD0_LES1 = {(SREG_BYTE_LEN-1){8'b0}};

typedef enum logic [3:0] {
    s_init,
    s_cmd_seq_start_wait,
    s_spi_port_request,
    s_spi_port_permit_wait,
    s_read_can_dev_sts,
    s_tx_empty_flag_clear,
    s_update_status,
    s_cmd_seq_done,

    s_spi_tx_proc,
    s_spi_tx_done_wait
} state_t;

//-----------------------------------------------------------
state_t curr_state, next_state, return_state;

logic   [(SREG_BYTE_LEN*8-1):0] spi_rx_data;
logic   [(SREG_BYTE_LEN*8-1):0] spi_tx_command;

logic   [7:0] total_byte_num;

logic   spi_tx_start;
logic   spi_tx_done;

can_dev_sts_t status_flags;

//-----------------------------------------------------------

shift_reg  #(
    .BYTE_LEN    (SREG_BYTE_LEN)
) 
spi_data_reg (
    .* ,
    .data_load      (spi_tx_start),
    .data_in        (spi_tx_command),
    .data_out       (spi_rx_data),
    .shift_carry    (cont_out.shift_carry),
    .shift_in       (cont_out.miso),
    .shift_out      (cont_out.mosi)
);

assign  cont_out.byte_num = total_byte_num;
assign  cont_out.start    = spi_tx_start;
assign  spi_tx_done = cont_out.done; 




assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        next_state <= s_init;
        return_state <= s_init;
        cont_out.request <= 1'b0;
        status_flags <= 8'b0; // これは一度だけ初期化する。
        total_byte_num <= 8'd0;
        spi_tx_start <= 1'b0;
        spi_tx_command <= PAD0_FULL;
    end
    else begin
        case ( curr_state )
            s_init: begin
                cont_out.request <= 1'b0;
                next_state <= s_cmd_seq_start_wait;
                return_state <= s_init;
                total_byte_num <= 8'd0;
                spi_tx_start <= 1'b0;
                spi_tx_command <= PAD0_FULL;
            end

            // SPI port の制御権を取得したら、Trigger がかかる。
            s_cmd_seq_start_wait: begin
                if ( start ) begin
                    next_state <= s_spi_port_request;
                end
            end

            // SPI port の制御権を要求する。
            s_spi_port_request: begin
                cont_out.request <= 1'b1;
                next_state <= s_spi_port_permit_wait;
            end

            // SPI port の制御権取得を待つ。
            s_spi_port_permit_wait: begin
                if ( cont_out.permit ) begin
                    next_state <= s_read_can_dev_sts;
                end
            end

            // 状態読み込み命令を発行する。
            s_read_can_dev_sts: begin
                total_byte_num <= 8'd2;
                spi_tx_command <= {
                    CMD_READ_STATUS, 
                    PAD0_LES1
                };
                next_state   <= s_spi_tx_proc;
                return_state <= s_tx_empty_flag_clear;
            end

            // TX EMPTY フラグが立っていたらクリアする。
            s_tx_empty_flag_clear: begin
                if ((spi_rx_data[7:0] & MASK_STATUS_TX012IF) == 8'b00) begin
                    next_state <= s_update_status;
                end
                else begin
                    total_byte_num <= 8'd4;
                    spi_tx_command <= {
                        CMD_BIT_MODIFY, 
                        ADRS_CANINTF, 
                        (CANINTF_TX0IF | CANINTF_TX1IF | CANINTF_TX2IF), 
                        8'b0
                    };
                    next_state   <= s_spi_tx_proc;
                    return_state <= s_read_can_dev_sts; // 再度、状態をリードする。
                end
            end

            // リードしたデータで 状態レジスタを更新する。
            s_update_status: begin
                status_flags <= spi_rx_data[7:0]; 
                next_state <= s_cmd_seq_done;
            end

            // コマンド制御の終了(cont cmd_seq_done)を出力する。
            s_cmd_seq_done: begin
                next_state <= s_init;
            end


            //----------------------------------------------------
            // SPI 通信を行う処理。 終了後に return する。
            s_spi_tx_proc: begin
                // spi へ trig をかける
                spi_tx_start <= 1'b1;
                next_state <= s_spi_tx_done_wait; 
            end
            // SPI制御の終了(spi done)を待つ。 
            s_spi_tx_done_wait: begin
                spi_tx_start <= 1'b0;
                if ( spi_tx_done ) begin
                    next_state <= return_state;
                end
            end
            //----------------------------------------------------


            default: begin
                next_state <= s_init;
            end
        endcase
    end
end

assign  can_dev_sts = status_flags;

assign  txbuf_busy[0] = status_flags.TXB0CNTRL_TXREQ;
assign  txbuf_busy[1] = status_flags.TXB1CNTRL_TXREQ;
assign  txbuf_busy[2] = status_flags.TXB2CNTRL_TXREQ;



always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        done <= 1'b0; 
    end
    else begin
        if (curr_state == s_cmd_seq_done)
            done <= 1'b1;
        else
            done <= 1'b0; 
    end
end




endmodule

