//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

`include "can_device_status.svh"

module  can_rcv_msg_read (
    input   wire            reset_n,
    input   wire            ck,
    input   wire            [7:0] stn_id,
    input   can_dev_sts_t   can_dev_sts,
    rcv_data_if.out         rcv_data_out,
    spi_cont_if.out         cont_out
);

`include "MCP2515_types.svh"
`include "MCP2515.svh"
`include "can_msg.svh"

//-----------------------------------------------------------
localparam  SREG_BYTE_LEN = 14;
localparam  PAD0_FULL = {(SREG_BYTE_LEN){8'b0}};
localparam  PAD0_LES1 = {(SREG_BYTE_LEN-1){8'b0}};

typedef enum logic [3:0] {
    s_init,
    s_cmd_seq_start_wait,
    s_rxb0_cmd_setup,
    s_rxb1_cmd_setup,
    s_spi_port_request,
    s_spi_port_permit_wait,
    s_rcv_msg_read,
    s_rcv_msg_store,
    s_spi_port_release,
    s_spi_port_release_wait,
    s_cmd_seq_done,

    s_spi_tx_proc,
    s_spi_tx_done_wait
} state_t;


//-----------------------------------------------------------
state_t curr_state, next_state, return_state;

logic   rxb0_read_start;
logic   rxb1_read_start;
logic   rxb0_read_start_clear;
logic   rxb1_read_start_clear;

logic   [(SREG_BYTE_LEN*8-1):0] spi_rx_data;
logic   [(SREG_BYTE_LEN*8-1):0] spi_tx_command;

logic   [7:0] total_byte_num;

logic   spi_tx_start;
logic   spi_tx_done;

can_rx_buf_t  can_rx_buf;

logic   [7:0] read_rx_buf_cmd;

logic   done;


//-----------------------------------------------------------
shift_reg  #(
    .BYTE_LEN    (SREG_BYTE_LEN)
) 
spi_data_reg (
    .* ,
    .data_load      (spi_tx_start),
    .data_in        (spi_tx_command),
    .data_out       (spi_rx_data),
    .shift_carry    (cont_out.shift_carry),
    .shift_in       (cont_out.miso),
    .shift_out      (cont_out.mosi)
);
assign  cont_out.byte_num = total_byte_num;
assign  cont_out.start    = spi_tx_start;
assign  spi_tx_done = cont_out.done; 


rise_latch   rxb0_start_det (
    .* ,
    .in       (can_dev_sts.CANINTF_RX0IF),
    .clear    (rxb0_read_start_clear),
    .out      (rxb0_read_start)
);


rise_latch   rxb1_start_det (
    .* ,
    .in       (can_dev_sts.CANINTF_RX1IF),
    .clear    (rxb1_read_start_clear),
    .out      (rxb1_read_start)
);


//-----------------------------------------------------------

assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        cont_out.request <= 1'b0;
        total_byte_num <= 8'b0;
        read_rx_buf_cmd <= 8'b0;
        rxb0_read_start_clear <= 1'b0;
        rxb1_read_start_clear <= 1'b0;
        spi_tx_command <= PAD0_FULL;
        spi_tx_start <= 1'b0;
        return_state <= s_init;
        next_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                cont_out.request <= 1'b0;
                total_byte_num <= 8'b0;
                read_rx_buf_cmd <= 8'b0;
                rxb0_read_start_clear <= 1'b0;
                rxb1_read_start_clear <= 1'b0;
                spi_tx_command <= PAD0_FULL;
                spi_tx_start <= 1'b0;
                return_state <= s_init;
                next_state <= s_cmd_seq_start_wait;
            end

            // start 条件を待つ。
            s_cmd_seq_start_wait: begin
                if ( rxb0_read_start ) begin
                    rxb0_read_start_clear <= 1'b1;
                    next_state <= s_rxb0_cmd_setup;
                end
                else if ( rxb1_read_start ) begin
                    rxb1_read_start_clear <= 1'b1;
                    next_state <= s_rxb1_cmd_setup;
                end
            end

            s_rxb0_cmd_setup: begin
                read_rx_buf_cmd <= (
                    CMD_READ_RX_BUF | CMD_READ_RX_BUF_AT_RXB0_SIDH
                );
                rxb0_read_start_clear <= 1'b0;
                next_state <= s_spi_port_request;
            end

            s_rxb1_cmd_setup: begin
                read_rx_buf_cmd <= (
                    CMD_READ_RX_BUF | CMD_READ_RX_BUF_AT_RXB1_SIDH
                );
                rxb1_read_start_clear <= 1'b0;
                next_state <= s_spi_port_request;
            end

            // spi port の制御権を要求する。
            s_spi_port_request: begin
                cont_out.request <= 1'b1;
                next_state <= s_spi_port_permit_wait;
            end

            // spi port の制御権取得を待つ。
            s_spi_port_permit_wait: begin
                if ( cont_out.permit ) begin
                    next_state <= s_rcv_msg_read;
                end
            end

            //---  メッセージ受信処理 --------------------------------
            s_rcv_msg_read: begin
                total_byte_num <= 8'd14;
                spi_tx_command <= {read_rx_buf_cmd, PAD0_LES1};
                next_state <= s_spi_tx_proc;
                return_state <= s_rcv_msg_store;
            end

            s_rcv_msg_store: begin
                next_state <= s_spi_port_release;
            end

            //---------------------------------------------------------
            // spi port の制御権を解放する。
            s_spi_port_release: begin
                cont_out.request <= 1'b0;
                next_state <= s_spi_port_release_wait;
            end

            // spi port の制御権解放を待つ。
            s_spi_port_release_wait: begin
                if (cont_out.permit == 1'b0) begin
                    next_state <= s_cmd_seq_done;
                end
            end

            // コマンド制御の終了(done)を出力する。
            s_cmd_seq_done: begin
                next_state <= s_init;
            end


            //----------------------------------------------------
            // spi 通信を行う処理。 終了後に return する。
            s_spi_tx_proc: begin
                // spi へ trig をかける
                spi_tx_start <= 1'b1;
                next_state <= s_spi_tx_done_wait; 
            end
            // spi制御の終了(spi done)を待つ。 
            s_spi_tx_done_wait: begin
                spi_tx_start <= 1'b0;
                if ( spi_tx_done ) begin
                    next_state <= return_state;
                end
            end
            //----------------------------------------------------


            default: begin
                next_state <= s_init;
            end

        endcase
    end
end


always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        can_rx_buf <= '0;
    end
    else begin
        case ( curr_state )
            s_rcv_msg_store: begin
                can_rx_buf <= spi_rx_data[0 +: $bits(can_rx_buf)];
            end

            default: begin
                can_rx_buf <= can_rx_buf;
            end

        endcase
    end
end


always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        done <= 1'b0; 
    end
    else begin
        if (curr_state == s_cmd_seq_done)
            done <= 1'b1;
        else
            done <= 1'b0; 
    end
end


assign  rcv_data_out.rx_buf = can_rx_buf;
assign  rcv_data_out.trig   = done;



endmodule

