//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  config_data_manager #(
    parameter  BYTE_LEN = 0
) (
    input   wire    reset_n,
    input   wire    ck,
    input   wire    read_cmd,
    output  logic   read_done,
    input   wire    write_cmd,
    output  logic   write_done,
    input   wire    [BYTE_LEN*8-1:0] din,
    output  logic   [BYTE_LEN*8-1:0] dout,
    spi_if.master   spi_port
);

`include "assert_param.svh"
initial begin
    `assert_param(BYTE_LEN >= 1, BYTE_LEN)
end

//-----------------------------------------------------------
localparam  MSB = (BYTE_LEN*8) -1;
localparam  START_ADRS = 16'h0000;
localparam  HALF_OF_SCK_PRI = 10;   // 2MHz

typedef enum logic [3:0] {
    s_init,
    s_cmd_start_wait,
    s_fram_write,
    s_write_done_wait,
    s_fram_read,
    s_read_done_wait,
    s_done
} state_t;


typedef enum logic [1:0] {
    c_invalid_cmd,
    c_read_cmd,
    c_write_cmd
} command_t;


//-----------------------------------------------------------
state_t     curr_state, next_state;
command_t   cmd;

spi_cont_if spi_cont_bus();

logic   read, write, done;

logic   [MSB:0] fram_out;


//-----------------------------------------------------------

fram_read_write # (
    .BYTE_LEN   (BYTE_LEN)
)
config_data_read_write (
    .*,
    .read       (read),
    .write      (write),
    .done       (done),
    .adrs       (START_ADRS),
    .din        (din),
    .dout       (fram_out),
    .cont_out   (spi_cont_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
fram_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_port)
); 



assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        read <= 1'b0; write <= 1'b0;
        dout <= '0;
        cmd <= c_invalid_cmd;
        next_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                read <= 1'b0; write <= 1'b0;
                // dout <= '0;   dout は保持する。
                cmd <= c_invalid_cmd;
                next_state <= s_cmd_start_wait;
            end

            // SPI port の制御権を取得したら、Trigger がかかる。
            s_cmd_start_wait: begin
                if ( write_cmd ) begin
                    write <= 1'b1;
                    cmd <= c_write_cmd;
                    next_state <= s_fram_write;
                end
                else if ( read_cmd ) begin
                    read <= 1'b1;
                    cmd <= c_read_cmd;
                    next_state <= s_fram_read;
                end
            end

            // FRAM へ data を書き込む命令を発行する。
            s_fram_write: begin
                write <= 1'b1;
                next_state   <= s_write_done_wait;
            end

            s_write_done_wait: begin
                write <= 1'b0;
                if ( done ) begin
                    next_state <= s_fram_read;
                end
            end

            // FRAM から data を読み込む命令を発行する。
            s_fram_read: begin
                read <= 1'b1;
                next_state   <= s_read_done_wait;
            end

            s_read_done_wait: begin
                read <= 1'b0;
                if ( done ) begin
                    dout <= fram_out;
                    next_state <= s_done;
                end
            end

            // コマンド制御の終了(done)を出力する。
            s_done: begin
                next_state <= s_init;
            end


            default: begin
                next_state <= s_init;
            end
        endcase
    end
end



always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        read_done <= 1'b0; 
        write_done <= 1'b0; 
    end
    else if (curr_state == s_done) begin
        if (cmd == c_read_cmd)
            read_done <= 1'b1;
        else if (cmd == c_write_cmd)
            write_done <= 1'b1;
    end
    else begin
        read_done <= 1'b0; 
        write_done <= 1'b0; 
    end
end





endmodule

