//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  fram_read_write #(
    parameter  BYTE_LEN = 0
) (
    input   wire    reset_n,
    input   wire    ck,
    input   wire    read,
    input   wire    write,
    output  logic   done,
    input   wire    [15:0]  adrs,
    input   wire    [BYTE_LEN*8-1:0] din,
    output  logic   [BYTE_LEN*8-1:0] dout,
    spi_cont_if.out cont_out
);

`include "assert_param.svh"
initial begin
    `assert_param(BYTE_LEN >= 1, BYTE_LEN)
end

//-----------------------------------------------------------
`include "FM25L16B.svh"

localparam  SREG_BYTE_LEN = 3 + BYTE_LEN; // cmd(1), adrs(2), data(BYTE_LEN)
localparam  PAD0_FULL = {(SREG_BYTE_LEN){8'b0}};
localparam  PAD0_LES1 = {(SREG_BYTE_LEN-1){8'b0}};
localparam  PAD0_LES3 = {(SREG_BYTE_LEN-3){8'b0}};

localparam  CMD_READ_BYTE_NUM  = $bits({CMD_READ, adrs, dout}) / 8;
localparam  CMD_WREN_BYTE_NUM  = 1;
localparam  CMD_WRITE_BYTE_NUM = $bits({CMD_WRITE, adrs, din}) / 8;


typedef enum logic [3:0] {
    s_init,
    s_cmd_start_wait,
    s_spi_port_request,
    s_spi_port_permit_wait,
    s_fram_read,
    s_read_data_store,
    s_fram_write_enable,
    s_fram_write,
    s_done,
    s_spi_tx_proc,
    s_spi_tx_done_wait
} state_t;

//-----------------------------------------------------------
state_t curr_state, next_state, return_state;
state_t cmd_state;

logic   [(SREG_BYTE_LEN*8-1):0] spi_rx_data;
logic   [(SREG_BYTE_LEN*8-1):0] spi_tx_command;

logic   [7:0] total_byte_num;

logic   spi_tx_start;
logic   spi_tx_done;


//-----------------------------------------------------------

shift_reg  #(
    .BYTE_LEN    (SREG_BYTE_LEN)
) 
spi_data_reg (
    .* ,
    .data_load      (spi_tx_start),
    .data_in        (spi_tx_command),
    .data_out       (spi_rx_data),
    .shift_carry    (cont_out.shift_carry),
    .shift_in       (cont_out.miso),
    .shift_out      (cont_out.mosi)
);

assign  cont_out.byte_num = total_byte_num;
assign  cont_out.start    = spi_tx_start;
assign  spi_tx_done = cont_out.done; 



assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        cont_out.request <= 1'b0;
        dout <= '0;
        total_byte_num <= 8'd0;
        spi_tx_start <= 1'b0;
        spi_tx_command <= PAD0_FULL;
        return_state <= s_init; cmd_state <= s_init;
        next_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                cont_out.request <= 1'b0;
                //dout <= '0;  dout は保持する。
                total_byte_num <= 8'd0;
                spi_tx_start <= 1'b0;
                spi_tx_command <= PAD0_FULL;
                return_state <= s_init; cmd_state <= s_init;
                next_state <= s_cmd_start_wait;
            end

            // SPI port の制御権を取得したら、Trigger がかかる。
            s_cmd_start_wait: begin
                if ( write ) begin
                    cmd_state <= s_fram_write_enable;
                    next_state <= s_spi_port_request;
                end
                else if ( read ) begin
                    cmd_state <= s_fram_read;
                    next_state <= s_spi_port_request;
                end
            end

            // SPI port の制御権を要求する。
            s_spi_port_request: begin
                cont_out.request <= 1'b1;
                next_state <= s_spi_port_permit_wait;
            end

            // SPI port の制御権取得を待つ。
            s_spi_port_permit_wait: begin
                if ( cont_out.permit ) begin
                    next_state <= cmd_state;
                end
            end


            // FRAM から data を読み込む命令を発行する。
            s_fram_read: begin
                total_byte_num <= CMD_READ_BYTE_NUM[7:0];
                spi_tx_command <= {
                    CMD_READ, 
                    adrs,
                    PAD0_LES3
                };
                next_state   <= s_spi_tx_proc;
                return_state <= s_read_data_store;
            end

            s_read_data_store: begin
                dout <= spi_rx_data[0 +: $bits(dout)];  
                next_state   <= s_spi_tx_proc;
                return_state <= s_done;
            end


            // FRAM へ data を書き込む命令を発行する。
            s_fram_write_enable: begin
                total_byte_num <= CMD_WREN_BYTE_NUM[7:0];
                spi_tx_command <= {
                    CMD_WREN, 
                    PAD0_LES1
                };
                next_state   <= s_spi_tx_proc;
                return_state <= s_fram_write;
            end

            s_fram_write: begin
                total_byte_num <= CMD_WRITE_BYTE_NUM[7:0];
                spi_tx_command <= {
                    CMD_WRITE, 
                    adrs,
                    din
                };
                next_state   <= s_spi_tx_proc;
                return_state <= s_done;
            end


            // コマンド制御の終了(done)を出力する。
            s_done: begin
                next_state <= s_init;
            end


            //----------------------------------------------------
            // SPI 通信を行う処理。 終了後に return する。
            s_spi_tx_proc: begin
                // spi へ trig をかける
                spi_tx_start <= 1'b1;
                next_state <= s_spi_tx_done_wait; 
            end
            // SPI制御の終了(spi done)を待つ。 
            s_spi_tx_done_wait: begin
                spi_tx_start <= 1'b0;
                if ( spi_tx_done ) begin
                    next_state <= return_state;
                end
            end
            //----------------------------------------------------


            default: begin
                next_state <= s_init;
            end
        endcase
    end
end



always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        done <= 1'b0; 
    end
    else begin
        if (curr_state == s_done)
            done <= 1'b1;
        else
            done <= 1'b0; 
    end
end





endmodule


