//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  msg_data_send #(
    parameter   TXBUF_NUM = 0
) (
    input   wire    reset_n,
    input   wire    ck,

    input   wire    [2:0] txbuf_busy,
    output  logic   done,
    can_msg_if.in   can_msg_in,
    spi_cont_if.out cont_out
);

`include "assert_param.svh"
initial begin
    `assert_param((0<=TXBUF_NUM)&(TXBUF_NUM<=2), TXBUF_NUM)
end

`include "MCP2515_types.svh"
`include "MCP2515.svh"

// 必要なバイトサイズは送信バイト数は 14 : {command, tx_buf} 
localparam  SREG_BYTE_LEN = 14;
localparam  PAD0_FULL = {(SREG_BYTE_LEN){8'b0}};
localparam  PAD0_LES1 = {(SREG_BYTE_LEN-1){8'b0}};


typedef enum logic [3:0] {
    s_init,
    s_cmd_seq_start_wait,
    s_spi_port_request,
    s_spi_port_permit_wait,
    s_write_to_txbuf,
    s_request_to_send,
    s_spi_port_release,
    s_spi_port_release_wait,
    s_cmd_seq_done,

    s_spi_tx_proc,
    s_spi_tx_done_wait
} state_t;


//-----------------------------------------------------------
state_t curr_state, next_state, return_state;

logic   [(SREG_BYTE_LEN*8-1):0] spi_rx_data;
logic   [(SREG_BYTE_LEN*8-1):0] spi_tx_command;

logic   [7:0] total_byte_num;

logic   spi_tx_start;
logic   spi_tx_done;

logic   txb_busy;
logic   [7:0] load_tx_buf_command;
logic   [7:0] request_to_sned_command;

logic   port_request;

can_tx_buf_t    tx_buf;

logic   send_start;
logic   trig_latch_reset;


localparam  CMD_LOAD_TXBUF_BYTE_NUM  = 1 + ($bits(tx_buf) >> 3);
localparam  CMD_REQ_TO_SEND_BYTE_NUM = 1;

//-----------------------------------------------------------

// シーケンス動作中の trig は１度だけ受け付けて保留しておく。
sr_latch  a_sr_latch (
    .*,
    .set    (can_msg_in.trig),
    .rst    (trig_latch_reset),
    .out    (send_start)
);

assign  trig_latch_reset = (curr_state == s_spi_port_request);



// 使用する tx buffer に応じて設定する。 
generate
if (TXBUF_NUM == 2) begin
    assign  txb_busy = txbuf_busy[2]; 
    assign  load_tx_buf_command = CMD_LOAD_TX_BUF_AT_TXB2_SIDH;
    assign  request_to_sned_command = CMD_REQ_TO_SEND_TXB2;
end
else if (TXBUF_NUM == 1) begin
    assign  txb_busy = txbuf_busy[1]; 
    assign  load_tx_buf_command = CMD_LOAD_TX_BUF_AT_TXB1_SIDH;
    assign  request_to_sned_command = CMD_REQ_TO_SEND_TXB1;
end
else if (TXBUF_NUM == 0) begin
    assign  txb_busy = txbuf_busy[0]; 
    assign  load_tx_buf_command = CMD_LOAD_TX_BUF_AT_TXB0_SIDH;
    assign  request_to_sned_command = CMD_REQ_TO_SEND_TXB0;
end
endgenerate


can_msg_to_tx_buf  a_can_msg_to_tx_buf (
    .can_msg_in         (can_msg_in.msg),
    .tx_buf_out         (tx_buf),
    .is_ext_data_frame  ()
);



shift_reg  #(
    .BYTE_LEN    (SREG_BYTE_LEN)
) 
spi_data_reg (
    .* ,
    .data_load      (spi_tx_start),
    .data_in        (spi_tx_command),
    .data_out       (spi_rx_data),
    .shift_carry    (cont_out.shift_carry),
    .shift_in       (cont_out.miso),
    .shift_out      (cont_out.mosi)
);
assign  cont_out.byte_num = total_byte_num;
assign  cont_out.start    = spi_tx_start;
assign  cont_out.request  = ~txb_busy & port_request;
assign  spi_tx_done = cont_out.done; 



assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        port_request <= 1'b0;
        total_byte_num <= 8'd0;
        spi_tx_start <= 1'b0;
        spi_tx_command <= PAD0_FULL;
        next_state <= s_init; 
        return_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                port_request <= 1'b0;
                total_byte_num <= 8'd0;
                spi_tx_start <= 1'b0;
                // spi_tx_command <= PAD0_FULL;   spi_tx_command は保持する。
                next_state <= s_cmd_seq_start_wait;
                return_state <= s_init;
            end

            // start 条件を待つ。
            s_cmd_seq_start_wait: begin
                if ( send_start ) begin
                    next_state <= s_spi_port_request;
                end
            end

            // spi port の制御権を要求する。
            s_spi_port_request: begin
                port_request <= 1'b1;
                next_state <= s_spi_port_permit_wait;
            end

            // spi port の制御権取得を待つ。
            s_spi_port_permit_wait: begin
                if ( cont_out.permit ) begin
                    next_state <= s_write_to_txbuf;
                end
            end

            s_write_to_txbuf: begin
                total_byte_num <= CMD_LOAD_TXBUF_BYTE_NUM[7:0];
                spi_tx_command <= {
                    load_tx_buf_command, 
                    tx_buf
                };
                next_state <= s_spi_tx_proc;
                return_state <= s_request_to_send;
            end

            s_request_to_send: begin
                total_byte_num <= CMD_REQ_TO_SEND_BYTE_NUM[7:0];
                spi_tx_command <= {
                    request_to_sned_command,
                    PAD0_LES1
                };
                next_state <= s_spi_tx_proc;
                return_state <= s_spi_port_release;
            end

            //---------------------------------------------------------
            // spi port の制御権を解放する。
            s_spi_port_release: begin
                port_request <= 1'b0;
                next_state <= s_spi_port_release_wait;
            end

            // spi port の制御権解放を待つ。
            s_spi_port_release_wait: begin
                if ( ~cont_out.permit ) begin
                    next_state <= s_cmd_seq_done;
                end
            end

            // コマンド制御の終了(done)を出力する。
            s_cmd_seq_done: begin
                next_state <= s_init;
            end



            //----------------------------------------------------
            // spi 通信を行う処理。 終了後に return する。
            s_spi_tx_proc: begin
                // spi へ trig をかける
                spi_tx_start <= 1'b1;
                next_state <= s_spi_tx_done_wait; 
            end
            // spi制御の終了(spi done)を待つ。 
            s_spi_tx_done_wait: begin
                spi_tx_start <= 1'b0;
                if ( spi_tx_done ) begin
                    next_state <= return_state;
                end
            end
            //----------------------------------------------------


            default: begin
                next_state <= s_init;
            end

        endcase
    end
end




always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        done <= 1'b0; 
    end
    else begin
        if (curr_state == s_cmd_seq_done)
            done <= 1'b1;
        else
            done <= 1'b0; 
    end
end



endmodule

