//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  spi_cont_switch #(
    parameter PORT_NUM = 0
) (
    input   wire    reset_n,
    input   wire    ck,

    input   wire    interrupt,
    output  logic   dev_sts_update,
    spi_cont_comb_if.in     cont_comb_in,
    spi_cont_if.out         cont_out
);

`include "assert_param.svh"
initial begin
    `assert_param(PORT_NUM > 0, PORT_NUM)
end


//-------------------------------------------------------------
typedef enum logic [1:0] {
    lc_after_reset,
    lc_normal_cmd,
    lc_can_dev_sts_read
} last_code_t;


typedef enum logic [3:0] {
    s_init,
    s_branch,
    s_request_check,
    s_int_or_request_wait,
    s_normal_cmd_start,
    s_dev_sts_update,
    s_can_dev_sts_read_request_wait,
    s_give_permission,
    s_permission_release_wait,
    s_cmd_done
} state_t;


//-----------------------------------------------------------
last_code_t last_code;
state_t     curr_state, next_state;

logic   port_request;
logic   port_permit;

logic   can_dev_sts_read_request;

//-------------------------------------------------------------
// SPI 制御権要求 (request) に応じて制御権を付与する。

spi_cont_sel #(
    .PORT_NUM   (PORT_NUM)
)
cont_sel (
    .*,
    .cont_comb_in   (cont_comb_in),
    .cont_sel       (cont_out),
    .port_request   (port_request),
    .port_permit    (port_permit)
);

// cont_comb_in.mo[1] に can_dev_sts_read_cont が割り付けられ
// ていることを前提としている。
assign  can_dev_sts_read_request = cont_comb_in.mo[1].request;


assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        port_permit <= 1'b0;
        last_code <= lc_after_reset;
        next_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                port_permit <= 1'b0;
                last_code <= lc_after_reset;
                next_state <= s_branch;
            end

            // s_branch 以降を繰り返す。
            s_branch: begin
                if (last_code == lc_can_dev_sts_read)
                    next_state <= s_request_check;
                else if (last_code == lc_normal_cmd)
                    next_state <= s_dev_sts_update;
                else
                    next_state <= s_int_or_request_wait;
            end

            s_request_check: begin
                if ( port_request )
                    next_state <= s_normal_cmd_start;
                else
                    next_state <= s_int_or_request_wait;
            end

            s_int_or_request_wait: begin
                if ( interrupt )
                    next_state <= s_dev_sts_update;
                else if ( port_request )
                    next_state <= s_normal_cmd_start;
            end

            // 通常のコマンドを start する。
            s_normal_cmd_start: begin
                last_code <= lc_normal_cmd;
                next_state <= s_give_permission;
            end

            // can_dev_sts_read を start する。
            s_dev_sts_update: begin
                next_state <= s_can_dev_sts_read_request_wait;
            end

            // can_dev_sts_read の request を待つ。
            s_can_dev_sts_read_request_wait: begin
                if ( can_dev_sts_read_request ) begin
                    last_code <= lc_can_dev_sts_read;
                    next_state <= s_give_permission;
                end
            end

            s_give_permission: begin
                port_permit <= 1'b1;
                next_state <= s_permission_release_wait;
            end

            s_permission_release_wait: begin
                if (port_request == 1'b0) begin
                    port_permit <= 1'b0;
                    next_state <= s_cmd_done;
                end
            end

            s_cmd_done: begin
                next_state <= s_branch;
            end

            default: begin
                next_state <= s_init;
            end

        endcase
    end
end



always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        dev_sts_update <= 1'b0;
    end
    else begin
        if(curr_state == s_dev_sts_update)
            dev_sts_update <= 1'b1;
        else
            dev_sts_update <= 1'b0;
    end
end






endmodule

