//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  spi_sck_counter #(
    parameter HALF_OF_SCK_PRI = 0
)(
    input   wire    reset_n,
    input   wire    ck,
    input   wire    [7:0] byte_num,
    input   wire    start,
    output  logic   shift_carry,
    output  logic   cs,
    output  logic   sck,
    output  logic   done
);

`include "assert_param.svh"
initial begin
    `assert_param(HALF_OF_SCK_PRI >= 2, HALF_OF_SCK_PRI)
end

// count 類の MSB を計算する。
localparam  MSB = $clog2( ((2**8)*8 +1)*2 );

typedef enum logic [2:0] {
    s_init,
    s_start_wait,
    s_chip_select,
    s_done_wait
} state_t;


//-----------------------------------------------------------
state_t curr_state, next_state;

logic   [MSB:0] count;
logic   [MSB:0] count_sck_stop;
logic   [MSB:0] count_sck_carry_stop;

logic   sck_carry_en;
logic   sck_carry;
logic   sck_stop;
logic   sck_carry_stop;

//-----------------------------------------------------------

carry_gen #(
    .INTERVAL   (HALF_OF_SCK_PRI)
) 
sck_carry_gen (
    .*,
    .en         (sck_carry_en),
    .carry      (sck_carry)
);


assign  curr_state = next_state;

always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        cs <= 1'b0; done <= 1'b0;
        next_state <= s_init;
    end
    else begin
        case ( curr_state )
            s_init: begin
                cs <= 1'b0; done <= 1'b0;
                next_state <= s_start_wait;
            end

            s_start_wait: begin
                if ( start ) begin
                    cs <= 1'b1;
                    next_state <= s_chip_select;
                end
            end

            s_chip_select: begin
                if ( sck_stop ) begin
                    cs <= 1'b0;
                    next_state <= s_done_wait;
                end
            end

            s_done_wait: begin
                if ( sck_carry_stop ) begin
                    done <= 1'b1;
                    next_state <= s_init;
                end
            end

            default: next_state <= s_init;
        endcase
    end
end


assign count_sck_stop = byte_num << 4;   // byte_num*8*2
assign count_sck_carry_stop = count_sck_stop + 2'd2;   // + idling time


always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n ) begin
        count <= 1'd1;
    end
    else begin
        case ( curr_state )
            s_init        : count <= 1'd1; 
            s_chip_select : if (sck_carry) count <= count + 1'd1;
            s_done_wait   : if (sck_carry) count <= count + 1'd1;
            default       : count <= count; 
        endcase
    end
end

assign  sck_carry_en   = (curr_state == s_chip_select) || (curr_state == s_done_wait);
assign  sck_stop       = sck_carry & (count == count_sck_stop);
assign  sck_carry_stop = sck_carry & (count == count_sck_carry_stop);



assign  shift_carry = sck_carry & sck;
assign  sck  = ~count[0] & cs;



endmodule

