//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------

class assertion_c;
    localparam  POS_EDGE = 0;
    localparam  NEG_EDGE = 1;
    localparam  OCCURR     = 1;
    localparam  NOT_OCCURR = 0;

    string  title;
    string  subtitle;

    event   _wait_cancel_ev;


    function new();
        this.title = "";
        this.subtitle = "";
    endfunction


    function void assert_equals(
            input   [(128*8-1):0] expected,
            input   [(128*8-1):0] actual,
            input   string format = "time",
            input   string unit = ""
        );

        real    real_exp, real_act;

        if (actual !== expected) begin
            $display("\n"); 
            $display("%s failed.", this.title); 
            $display("  %s", this.subtitle); 

            if (format == "time") begin
                $display("    expected = %t ", expected); 
                $display("    actual   = %t ", actual); 
            end
            else if (format == "int") begin
                /* 桁数に 1 を指定すると、最小の桁で表示する。(以下同様) */
                $display("    expected = %1d %s", expected, unit); 
                $display("    actual   = %1d %s", actual,   unit); 
            end
            else if (format == "hex") begin
                $display("    expected = 0x%1h", expected); 
                $display("    actual   = 0x%1h", actual); 
            end
            else if (format == "bin") begin
                $display("    expected = 0b%1b", expected); 
                $display("    actual   = 0b%1b", actual); 
            end
            else if (format == "real") begin
                real_exp = $bitstoreal( expected );
                real_act = $bitstoreal( actual );

                $display("    expected = %1f", real_exp); 
                $display("    actual   = %1f", real_act); 
            end
            else begin
                $display("    unknown format. : %s", format); 
            end

            $display("\n"); 

            $finish;
        end

    endfunction


    function void assert_true(
            input   logic actual
        );

        this.assert_equals(1'b1, actual, "bin");

    endfunction


    function void assert_range(
            input   [(128*8-1):0] expected_lower,
            input   [(128*8-1):0] expected_upper,
            input   [(128*8-1):0] actual,
            input   string format = "time",
            input   string unit = ""
        );

        logic in_range =  ((expected_lower <= actual) && (actual <= expected_upper));

        if ( ~in_range ) begin
            $display("\n"); 
            $display("%s failed.", this.title); 
            $display("  %s", this.subtitle); 

            if (format == "time") begin
                $display("    expected = %t ~ %t", expected_lower, expected_upper); 
                $display("    actual   = %t", actual); 
            end
            else if (format == "int") begin
                /* 桁数に 1 を指定すると、最小の桁で表示する。(以下同様) */
                $display("    expected = %1d ~ %1d %s", expected_lower, expected_upper, unit); 
                $display("    actual   = %1d %s", actual, unit); 
            end
            else if (format == "hex") begin
                $display("    expected = %1h ~ %1h %s", expected_lower, expected_upper, unit); 
                $display("    actual   = 0x%1h %s", actual, unit); 
            end
            else if (format == "bin") begin
                $display("    expected = 0b%1b ~ 0b%1b %s", expected_lower, expected_upper, unit); 
                $display("    actual   = 0b%1b", actual, unit); 
            end
            else begin
                $display("    unknown format. : %s", format); 
            end

            $display("\n"); 

            $finish;
        end

    endfunction


    task automatic assert_pos_edge(
            const ref logic signal,
            input time wait_time
        );

        string abort_msg = this._make_edge_detect_abort_message(POS_EDGE);

        fork : pos_edge_abort_detect
            this._abort_detect(wait_time, abort_msg);
        join_none

        @(posedge signal);
        disable pos_edge_abort_detect;

    endtask


    task automatic assert_neg_edge(
            const ref logic signal,
            input time wait_time
        );

        string abort_msg = this._make_edge_detect_abort_message(NEG_EDGE);

        fork : neg_edge_abort_detect
            this._abort_detect(wait_time, abort_msg);
        join_none

        @(negedge signal);
        disable neg_edge_abort_detect;

    endtask


    function string _make_edge_detect_abort_message(
            input int dir
        );

        string direction = (dir == POS_EDGE) ? "pos" : "neg";
        string msg = {this.title, " failed.  ", direction, " edge did not occurr."}; 

        return msg;

    endfunction


    task automatic assert_not_occurr(
            const ref logic signal,
            input time wait_time
        );

        int     result = OCCURR;
        string  msg;

        fork : wait_cancel_task
            this._wait_cancel(wait_time, result);
        join_none
    
        @(posedge signal or this._wait_cancel_ev);
        disable wait_cancel_task;

        if (result == OCCURR) begin
            msg = {this.title, " failed by signal event."}; 
            this._fail_display(msg);
            $finish;
        end

    endtask


    task automatic _wait_cancel(
            input time  wait_time,
            output int  ret
        );

        #(wait_time);

        ret = NOT_OCCURR;
        -> this._wait_cancel_ev;

    endtask

    task automatic _abort_detect(
            input time   wait_time,
            input string abort_msg
        );

        #(wait_time);

        this._fail_display(abort_msg);
        $finish;

    endtask


    function void _fail_display(
            input string message
        );

        $display("\n"); 
        $display("%s", message); 
        $display("  %s", this.subtitle); 
        $display("\n"); 

    endfunction


endclass : assertion_c

