//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
//  (1) reset 後は、出力は 0 であること。
//  (2) 0 to 1 変化して 1 ck 後に出力が 1 へ変化すること。
//  (3) 入力が 0  になっても出力 1 が保持されていること。
//  (4) clear 入力後 1ck 後に出力が 0 になること。
//
module   test006_01_rise_latch_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test006_01_rise_latch_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;

logic   actual;
logic   expected;
logic   signal;
logic   latch_clear;
logic   latch_out;


// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
localparam CARRY_INTERVAL = 40000;  // pri = ck * 4000 (1ms 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス
rise_latch   the_rise_latch (
    .* ,
    .in       (signal),
    .clear    (latch_clear),
    .out      (latch_out)
);


//-----------------------------------------------------------
assertion_c test;

initial begin
    test = new();

    test.title = title;
    reset_n = 1'b0;

    actual = 1'b0;
    expected = 1'b0;
    signal = 1'b0;
    latch_out = 1'b0;
    latch_clear = 1'b0;

    #300 reset_n = 1'b1;

    #(SIM_CLK_COUNT*100);


    //-----------------------------------------------------------------
    // (1) reset 後は、出力は 0 であること。
    test.subtitle = "(1) out is 0 after reset.";
    expected = 1'b0;
    actual = latch_out;
    test.assert_equals(expected, actual, "bin");

    @(posedge ck);

    //-----------------------------------------------------------------
    // (2) 0 to 1 変化して 1 ck 後に出力が 1 へ変化すること。
    signal = 1'b1;

    @(posedge ck);

    #2;
    test.subtitle = "(2) out is 1 after rise.";
    expected = 1'b1;
    actual = latch_out;
    test.assert_equals(expected, actual, "bin");


    //-----------------------------------------------------------------
    // (3) 入力が 0  になっても出力 1 が保持されていること。
    #(SIM_CLK_COUNT*50);
    signal = 1'b0;

    @(posedge ck);

    #2;
    test.subtitle = "(3) out is 1 after signal off.";
    expected = 1'b1;
    actual = latch_out;
    test.assert_equals(expected, actual, "bin");


    //-----------------------------------------------------------------
    // (4) clear 入力後 1ck 後に出力が 0 になること。
    #(SIM_CLK_COUNT*50);
    
    @(posedge ck);
    latch_clear = 1'b1;
    @(posedge ck);
    latch_clear = 1'b0;

    #2;
    test.subtitle = "(4) out is 0 after clear.";
    expected = 1'b0;
    actual = latch_out;
    test.assert_equals(expected, actual, "bin");




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end



endmodule

