//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) 入力信号を変化させ、出力に結果が現れるまでの時間を計測する。
// (2) 再度、入力信号を変化させ、出力に結果が現れるまでの時間を計測する。
//
module   test009_04_chattering_filter_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test009_04_chattering_filter_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;

logic   carry_gen_en;
logic   carry;


localparam  SIG_BIT_WIDTH = 8;
localparam  DURATION_CNT = 3'd5;

localparam  MSB = SIG_BIT_WIDTH - 1;
logic   [MSB:0] data_in;
logic   [MSB:0] data_out;
logic           update;

logic   [MSB:0] expected_data;
logic   [MSB:0] actual_data;

time    expected_time;
time    actual_time;


// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
//localparam CARRY_INTERVAL = 40000;  // pri = ck * 40000 (1ms 相当)
localparam CARRY_INTERVAL = 400;  // pri = ck * 400 (10us 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (CARRY_INTERVAL)
) 
a_carry_gen (
    .ck         (ck),
    .reset_n    (reset_n),
    .en         (carry_gen_en),
    .carry      (carry)
);


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス

chattering_filter #(
    .SIG_BIT_WIDTH      ($bits(data_in)),
    .CNT_BIT_WIDTH      ($bits(DURATION_CNT))
) 
the_chattering_filter (
    .reset_n   (reset_n),
    .ck        (ck),
    .carry     (carry),
    .duration  (DURATION_CNT),

    .data_in   (data_in),
    .data_out  (data_out),
    .update    (update)
);


//-----------------------------------------------------------
logic   data_is_match;

assertion_c test;
timer_c     tim1;

assign  data_is_match = (data_in == data_out);

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;
    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    data_in = 8'h00;
    timeout = 0;
    actual_time = 0; expected_time = 0;
    actual_data = '0; expected_data = '0;


    #100 reset_n = 1'b1;
    #200 carry_gen_en = 1'b1;


    //-------------------------------------------------------------
    // (1) 入力信号を変化させ、出力に結果が現れるまでの時間を計測する。
    @(posedge carry);
    @(posedge carry);


    // data_in を変化させる。
    @(negedge carry);   // negedge
    data_in = 8'hab;
    tim1.start();
    timeout = (DURATION_CNT * CARRY_INTERVAL * SIM_CLK_COUNT) + (SIM_CLK_COUNT*2);

    test.subtitle = "(1.2) the output change wait.";
    test.assert_pos_edge(data_is_match, timeout);

    test.subtitle = "(1.2) data change detect time test.";
    expected_time = (DURATION_CNT * CARRY_INTERVAL * SIM_CLK_COUNT);
    expected_time = expected_time + SIM_CLK_COUNT;  // 1 CLK 遅れる。
    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");


    //-------------------------------------------------------------
    // (2) 再度、入力信号を変化させ、出力に結果が現れるまでの時間を計測する。
    
    // data_in を変化させる。
    @(negedge carry);   // negedge

    expected_data = data_in; // 前回値を保存する。
    data_in = 8'hcd;
    tim1.start();
    timeout = (DURATION_CNT * CARRY_INTERVAL * SIM_CLK_COUNT) + (SIM_CLK_COUNT*2);

    #SIM_CLK_COUNT;
    actual_data = data_out;
    test.subtitle = "(2.1) data is not change.";
    test.assert_equals(expected_data, actual_data, "hex");


    test.subtitle = "(2.2) the output change wait.";
    test.assert_pos_edge(data_is_match, timeout);

    test.subtitle = "(2.3) data change detect time test.";
    expected_time = (DURATION_CNT * CARRY_INTERVAL * SIM_CLK_COUNT);
    expected_time = expected_time + SIM_CLK_COUNT;  // 1 CLK 遅れる。
    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");

    expected_data = 8'hcd;
    actual_data = data_out;
    test.subtitle = "(2.4) data is 8'hcd.";
    test.assert_equals(expected_data, actual_data, "hex");



    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end




endmodule

