//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// sck (6.666... MHz 設定時) の ON 時間、OFF 時間の確認を行う。
//  (1) reset 後 sck は、 1'b0 である。
//  (2) start から sck 立ち上がりまでの時間を計測する。
//  (3) sck が ON の時間を計測する。
//
module   test102_02_spi_cont_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test102_02_spi_cont_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;
time    expected_time;
time    actual_time;
logic   expected_level;
logic   actual_level;

// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 
localparam   HALF_OF_SCK_PRI = 3;
localparam   SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:6.666... MHz 
localparam   BYTE_LEN = 4;

logic   sck;
spi_cont_if     spi_cont_bus();
spi_if          spi_bus();
                             
spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
the_spi_cont (
    .ck         (ck),
    .reset_n    (reset_n),
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;



//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    spi_cont_bus.start = 1'b0;
    spi_cont_bus.request = 1'b0;
    spi_cont_bus.byte_num = BYTE_LEN;
    actual_time = 0; expected_time = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    timeout = 0;


    #100  reset_n = 1'b1;

    @(posedge ck);

    //-------------------------------------------------------------
    //  (1) reset 後 sck は、 1'b0 である。
    test.subtitle = "(1) sck is 1'b0 just after reset.";
    expected_level = 1'b0;
    actual_level = sck;
    test.assert_equals(expected_level, actual_level, "bin");



    //-------------------------------------------------------------
    //  (2) start から sck 立ち上がりまでの時間を計測する。
    test.subtitle = "(2.1) sck rise time test. : wait for sck rise.";
    expected_time = (SCK_PRI / 2) * SIM_CLK_COUNT;
    timeout = expected_time + SIM_COUNT_1US; 

    #1 spi_cont_bus.start = 1'b1;
    @(posedge ck);
    tim1.start();
    #1 spi_cont_bus.start = 1'b0;

    test.assert_pos_edge(sck, timeout);

    test.subtitle = "(2.2) sck rise time test.";
    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");

    //-------------------------------------------------------------
    //  (3) sck が ON の時間を計測する。
    test.subtitle = "(3) sck on time test.";

    test.assert_neg_edge(sck, timeout);

    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

