//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// can_device_config の動作を確認する。
//   (1) start 検出後、SPI port 制御権を取得する。 
//   (2) RESET コマンドを発行する。
//   (3) RESET 後 5ms 待つ。
//   (4) 基本設定(CNF1~3, INT en)を行う。
//   (5) 受信バッファの構成を設定する。
//   (6) 通常モードに設定し、CAN 通信を開始する。
//   (7) done が出力されることを確認する。
//
module   test103_01_can_device_config_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

localparam  title = "\"test103_01_can_device_config_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"


`include "MCP2515_types.svh"
`include "MCP2515.svh"
`include "can_device_status.svh"

`define  BIT_RATE_1M
`include "sim_bit_rate_param.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定
localparam CARRY_INTERVAL = 40000;  // pri = ck * 40000 (1ms 相当)

// 使用するコマンドの最大サイズ (=6)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 

localparam  WAIT_AFTER_RESET_MS = 20;

//-----------------------------------------------------------
logic   ck, reset_n;
logic   carry_1ms, carry_gen_en;
time    timeout;
time    expected_time,  actual_time;
time    exp_time_lower, exp_time_upper;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;

logic   cont_start;
logic   cont_done;

CNF1_t      CNF1;
CNF2_t      CNF2;
CNF3_t      CNF3;
CANINTE_t   CANINTE;
RXB0CTRL_t  RXB0CTRL;
CANCTRL_t   CANCTRL;



//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (CARRY_INTERVAL)
) 
a_carry_gen (
    .*,
    .en         (carry_gen_en),
    .carry      (carry_1ms)
);


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

can_device_config   the_can_device_config (
    .ck         (ck),
    .reset_n    (reset_n),
    .carry_1ms  (carry_1ms),
    .start      (cont_start),
    .cont_out   (spi_cont_bus),
    .done       (cont_done)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    cont_start = 1'b0;
    CNF1 = 8'b0; CNF2 = 8'b0; CNF3 = 8'b0;
    CANINTE = 8'b0; RXB0CTRL = 8'b0; CANCTRL = 8'b0;
    actual_time = 0; expected_time = 0;
    exp_time_lower = 0; exp_time_upper = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #100  carry_gen_en = 1'b1;
    #500;

    @(posedge ck);

    //-----------------------------------------------------------------
    // (1) start 検出後、SPI port 制御権を取得する。 

    // trig がかかる前は、 request は off 。
    test.subtitle = "(1.1) request is off before start.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");


    //-----------------------------------------------------------------
    // (2) RESET コマンドを発行する。
    a_sim_spi_slave.cmd_bit_len = $bits(CMD_RESET);
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // start をかける。
    @(posedge ck) cont_start = 1'b1;
    @(posedge ck) cont_start = 1'b0;


    // テスト対象のモジュールから request が出力されることを確認する。
    test.subtitle = "(2.1) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);


    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(2.2) SPI com (RESET command) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);


    test.subtitle = "(2.3) RESET command test.";
    expected_data = CMD_RESET;
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (3) RESET 後 20ms 待つ。
    test.subtitle = "(3.1) wait 20ms after RESET command.";
    tim1.start();
    timeout = (WAIT_AFTER_RESET_MS+1) * SIM_COUNT_1MS;
    test.assert_pos_edge(the_can_device_config.wait_time_up, timeout);

    exp_time_lower = (WAIT_AFTER_RESET_MS-1) * SIM_COUNT_1MS; 
    exp_time_upper = (WAIT_AFTER_RESET_MS+1) * SIM_COUNT_1MS; 
    actual_time = tim1.lap_time();

    test.subtitle = "(3.2) wait time 4 ~ 5 ms.";
    test.assert_range(exp_time_lower, exp_time_upper, actual_time, "time"); 
    //dspl_value("wait after reset time", actual_time, "time");


    //-----------------------------------------------------------------
    // (4) 基本設定(CNF1~3, INT en)を行う。
    a_sim_spi_slave.cmd_bit_len = 6 * 8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(4.1) write CNF3,2,1, CANINTE register test.: spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);


    test.subtitle = "(4.2) write CNF3,2,1, CANINTE register test.";

    CNF3 = LP_CNF3; CNF2 = LP_CNF2; CNF1 = LP_CNF1; 
    CANINTE = 8'b0;
    CANINTE.TX2IE = 1'b1; CANINTE.TX1IE = 1'b1; CANINTE.TX0IE = 1'b1;
    CANINTE.RX1IE = 1'b1; CANINTE.RX0IE = 1'b1;

    expected_data = {CMD_WRITE, ADRS_CNF3, CNF3, CNF2, CNF1, CANINTE};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (5) 受信バッファの構成を設定する。
    a_sim_spi_slave.cmd_bit_len = 4 * 8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    test.subtitle = "(5.1) bit modify RXB0CTRL register test.: spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);
    
    
    test.subtitle = "(5.2) bit modify RXB0CTRL register test.";

    RXB0CTRL.BUKT = 1'b1;
    expected_data = { 
        CMD_BIT_MODIFY,
        ADRS_RXB0CTRL,
        RXB0CTRL,   // as mask
        RXB0CTRL    // as data
    };
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (6) 通常モードに設定し、CAN 通信を開始する。
    a_sim_spi_slave.cmd_bit_len = 4 * 8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    test.subtitle = "(6.1) bit modify CANCTRL register test.: spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);
    
    
    test.subtitle = "(6.2) bit modify CANCTRL register test.";

    CANCTRL.REQOP = REQOP_NORMAL;
    expected_data = {
        CMD_BIT_MODIFY,
        ADRS_CANCTRL,
        MASK_CAN_CTRL_REQOP,    // as mask
        CANCTRL                 // as data
    };
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (7) done が出力されることを確認する。

    // can_device_config からの request が off になることを確認する。
    test.subtitle = "(7.1) request off wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(spi_cont_bus.request, timeout);

    // 完了後 cont_done が出力される。
    test.subtitle = "(7.2) cont_done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cont_done, timeout);




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

