//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// txbuf1 の empty 割り込みフラグをクリアする動作を確認する。
//   (1) can_dev_sts(MCP2515 STATUS 読み込み)リード動作を確認する。 
//   (2) 割り込みフラグのクリア動作を確認する。
//   (3) 再度 can_dev_sts リードで割り込みフラグがクリアされたことを
//       確認し、処理を完了することを確認する。
//   (4) done が出力されることを確認する。
//
module   test104_02_can_device_status_read_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

localparam  title = "\"test104_02_can_device_status_read_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"

`include "MCP2515.svh"
`include "can_device_status.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

// 使用するコマンドの最大サイズ (=4)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
time    expected_time,  actual_time;
time    exp_time_lower, exp_time_upper;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;

logic   cont_start;
logic   cont_done;

can_dev_sts_t   can_dev_sts;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

can_device_status_read     the_can_device_status_read (
    .ck             (ck),
    .reset_n        (reset_n),
    .start          (cont_start),
    .done           (cont_done),
    .can_dev_sts    (can_dev_sts),
    .cont_out       (spi_cont_bus)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    cont_start = 1'b0;
    actual_time = 0; expected_time = 0;
    exp_time_lower = 0; exp_time_upper = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------
    // (1) can_dev_sts(MCP2515 STATUS 読み込み)リード動作を確認する。 

    // (1.1) start がかかる前は、 request は off 。
    test.subtitle = "(1.1) request is off before start.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");

    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = $bits(CMD_READ_STATUS);
    a_sim_spi_slave.res_bit_len = $bits(STATUS_CANINTF_TX1IF);
    a_sim_spi_slave.response    = STATUS_CANINTF_TX1IF;

    // start をかける。
    @(posedge ck) cont_start = 1'b1;
    @(posedge ck) cont_start = 1'b0;

    // (1.2) request が出力されることを確認する。
    test.subtitle = "(1.2) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(1.3) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(1.4) 1st READ STATUS data check.";
    expected_data = CMD_READ_STATUS;
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (2) 割り込みフラグのクリア動作を確認する。
    a_sim_spi_slave.cmd_bit_len = 4 * 8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(2.1) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(2.2) empty flag clear test.";
    expected_data = {
        CMD_BIT_MODIFY, ADRS_CANINTF, 
        (CANINTF_TX0IF | CANINTF_TX1IF | CANINTF_TX2IF), 8'b0
    };
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (3) 再度 can_dev_sts リードで割り込みフラグがクリアされたことを
    //     確認し、処理を完了することを確認する。
    a_sim_spi_slave.cmd_bit_len = $bits(CMD_READ_STATUS);
    a_sim_spi_slave.res_bit_len = $bits(STATUS_CANINTF_TX1IF);
    a_sim_spi_slave.response    = STATUS_CANINTF_TX1IF & 8'b0;

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(3.1) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(3.2) 2nd READ STATUS data check.";
    expected_data = CMD_READ_STATUS;
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (4) done が出力されることを確認する。

    // 完了後 done が出力される。
    test.subtitle = "(4) can_device_status_read cont_done wait.";
    timeout = SIM_CLK_COUNT * 6;
    test.assert_pos_edge(cont_done, timeout);




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

