//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// 読み込んだ情報が can_dev_sts へ出力されることを確認する。
//
module   test104_04_can_device_status_read_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

localparam  title = "\"test104_04_can_device_status_read_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"

`include "MCP2515.svh"
`include "can_device_status.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

// 使用するコマンドの最大サイズ (=4)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
time    expected_time,  actual_time;
time    exp_time_lower, exp_time_upper;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;

logic   cont_start;
logic   cont_done;

can_dev_sts_t   can_dev_sts;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

can_device_status_read     the_can_device_status_read (
    .ck             (ck),
    .reset_n        (reset_n),
    .start          (cont_start),
    .done           (cont_done),
    .can_dev_sts    (can_dev_sts),
    .cont_out       (spi_cont_bus)
);


//-----------------------------------------------------------

task read_can_device_status;
    input   [7:0] expected_ans;

    string  _subtitle;

    begin
        // spi slave インスタンスの設定
        a_sim_spi_slave.cmd_bit_len = $bits(CMD_READ_STATUS);
        a_sim_spi_slave.res_bit_len = $bits(expected_ans);
        a_sim_spi_slave.response    = expected_ans;

        // start をかける。
        @(posedge ck) cont_start = 1'b1;
        @(posedge ck) cont_start = 1'b0;

        // 処理の完了を待つ。
        _subtitle = test.subtitle;
        test.subtitle = {_subtitle, " : cont_done wait."};

        timeout = SIM_COUNT_1MS;
        test.assert_pos_edge(cont_done, timeout);

        test.subtitle = _subtitle;

        #(SIM_CLK_COUNT * 100);
    end

endtask


assertion_c test;
timer_c     tim1;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    cont_start = 1'b0;
    actual_time = 0; expected_time = 0;
    exp_time_lower = 0; exp_time_upper = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    /*--- 1. TXB2CNTRL_TXREQ の出力動作を確認する。 -----*/
    test.subtitle = "(1.1) TXB2CNTRL_TXREQ(off) test.";
    read_can_device_status( 8'b0 );
    test.assert_equals(1'b0, can_dev_sts.TXB2CNTRL_TXREQ, "bin");

    test.subtitle = "(1.2) TXB2CNTRL_TXREQ test.";
    read_can_device_status( STATUS_TXB2CNTRL_TXREQ );
    test.assert_equals(1'b1, can_dev_sts.TXB2CNTRL_TXREQ, "bin");


    /*--- 2. TXB1CNTRL_TXREQ の出力動作を確認する。 -----*/
    test.subtitle = "(2.1) TXB1CNTRL_TXREQ(off) test.";
    read_can_device_status( 8'b0 );
    test.assert_equals(1'b0, can_dev_sts.TXB1CNTRL_TXREQ, "bin");

    test.subtitle = "(2.2) TXB1CNTRL_TXREQ test.";
    read_can_device_status( STATUS_TXB1CNTRL_TXREQ );
    test.assert_equals(1'b1, can_dev_sts.TXB1CNTRL_TXREQ, "bin");


    /*--- 3. TXB0CNTRL_TXREQ の出力動作を確認する。 -----*/
    test.subtitle = "(3.1) TXB0CNTRL_TXREQ(off) test.";
    read_can_device_status( 8'b0 );
    test.assert_equals(1'b0, can_dev_sts.TXB0CNTRL_TXREQ, "bin");

    test.subtitle = "(3.2) TXB0CNTRL_TXREQ test.";
    read_can_device_status( STATUS_TXB0CNTRL_TXREQ );
    test.assert_equals(1'b1, can_dev_sts.TXB0CNTRL_TXREQ, "bin");


    /*--- 4. CANINTF_RX1IF   の出力動作を確認する。 -----*/
    test.subtitle = "(4.1) CANINTF_RX1IF(off) test.";
    read_can_device_status( 8'b0 );
    test.assert_equals(1'b0, can_dev_sts.CANINTF_RX1IF, "bin");

    test.subtitle = "(4.2) CANINTF_RX1IF test.";
    read_can_device_status( STATUS_CANINTF_RX1IF );
    test.assert_equals(1'b1, can_dev_sts.CANINTF_RX1IF, "bin");


    /*--- 5. CANINTF_RX0IF   の出力動作を確認する。 -----*/
    test.subtitle = "(5.1) CANINTF_RX0IF(off) test.";
    read_can_device_status( 8'b0 );
    test.assert_equals(1'b0, can_dev_sts.CANINTF_RX0IF, "bin");

    test.subtitle = "(5.2) CANINTF_RX0IF test.";
    read_can_device_status( STATUS_CANINTF_RX0IF );
    test.assert_equals(1'b1, can_dev_sts.CANINTF_RX0IF, "bin");




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end


endmodule

