//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// can_rcv_msg_read の動作を確認する。
//  (1) 受信割り込み(RXB0)で SPI 通信が開始する。
//  (2) 取り込んだ受信メッセージを確認する。
//  (3) RXB0 read の start 条件が off になっていることを確認する。
//      (off になっていないと不正な read 命令になってしまう。)
//
module   test105_01_can_rcv_msg_read_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test105_01_can_rcv_msg_read_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"

`include "MCP2515_types.svh"
`include "MCP2515.svh"
`include "can_device_status.svh"
`include "can_msg.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  CONT_STN_ID = 8'h03;
localparam  DEST_STN_ID = 8'h05;

// 使用するコマンドの最大サイズ (=4)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;

logic   cont_start;
logic   cont_done;

can_dev_sts_t   can_dev_sts;

rcv_data_if     rcv_data_out();
can_rx_buf_t    expected_rx_buf, actual_rx_buf;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

can_rcv_msg_read     the_can_rcv_msg_read (
    .ck             (ck),
    .reset_n        (reset_n),

    .stn_id         (DEST_STN_ID),
    .can_dev_sts    (can_dev_sts),

    .rcv_data_out   (rcv_data_out),
    .cont_out       (spi_cont_bus)
);

assign  cont_done = rcv_data_out.trig;


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    cont_start = 1'b0;
    can_dev_sts = '0;
    expected_rx_buf = '0;
    actual_rx_buf = '0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------

    // 受信メッセージの準備を行う。
    {expected_rx_buf.STD_ID, expected_rx_buf.EXT_ID} = {
        1'b0, CONT_STN_ID, 4'b0, MSG_CONFIG_DATA_WRITE, DEST_STN_ID
    };
    expected_rx_buf.IDE          = 1'b1;  // ext id で運用する。
    expected_rx_buf.DLC[3:0]     = 4'd8;
    expected_rx_buf.DATA_BYTE    = 64'h0123456789ABCDEF;

    //-----------------------------------------------------------------
    // (1) 受信割り込みで SPI 通信が開始することを確認する。

    // (1.1) 割り込みがかかる前は、 request は off 。
    test.subtitle = "(1.1) request is off before interrupt.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");

    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = $bits(CMD_READ_RX_BUF);
    a_sim_spi_slave.res_bit_len = $bits(expected_rx_buf);
    a_sim_spi_slave.response    = expected_rx_buf;

    // rx buf 0 の受信割り込みを発生する。
    can_dev_sts.CANINTF_RX0IF = 1'b1;

    // (1.2) request が出力されることを確認する。
    test.subtitle = "(1.2) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);

    // (1.3) SPI 通信の完了を待つ。
    test.subtitle = "(1.3) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    // (1.4) 受信した command データと期待値を照合する。
    test.subtitle = "(1.4) READ RX BUF command data test.";
    expected_data = CMD_READ_RX_BUF | CMD_READ_RX_BUF_AT_RXB0_SIDH;
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    //  (2) 取り込んだ CAN 受信メッセージ を確認する。

    // (2.1) 完了後 done が出力される。
    test.subtitle = "(2.1) can_rcv_msg_read cont_done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cont_done, timeout);

    // (2.2) 受信した CAN 受信メッセージ と期待値を照合する。
    test.subtitle = "(2.2) READ RX BUF command data test.";
    actual_rx_buf = rcv_data_out.rx_buf;
    test.assert_equals(expected_rx_buf, actual_rx_buf, "hex");


    //-----------------------------------------------------------------
    // (3) RXB0 read の start 条件が off になっていることを確認する。
    //     (off になっていないと不正な read 命令になってしまう。)
    test.subtitle = "(3) rxb0 read start condition is off after done.";
    expected_level = 1'b0;
    actual_level = the_can_rcv_msg_read.rxb0_read_start;
    test.assert_equals(expected_level, actual_level, "bin");





    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

