//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// destination_filter の動作を確認する。
// (1) (全局宛)  message に対する動作を確認する。
// (2) (自局宛)  message に対する動作を確認する。
// (3) (他局宛)  message に対する動作を確認する。
//
module   test106_01_destination_filter_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test106_01_destination_filter_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"

`include "MCP2515_types.svh"
`include "can_msg_types.svh"

`include "MCP2515.svh"
`include "can_device_status.svh"
`include "can_msg.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  ALL_STATION  = 8'h00;
localparam  CONT_STN_ID  = 8'h03;
localparam  DEST_STN_ID  = 8'h05;
localparam  OTHER_STN_ID = 8'h12;


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
logic   expected_level, actual_level;

can_rx_buf_t    test_can_rx_buf;
can_msg_t       test_can_msg;
can_msg_t       expected_msg, actual_msg;

logic       [7:0] stn_id;   
rcv_data_if rcv_data_bus();
can_msg_if  can_msg_bus();

logic   trig;

//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


rx_buf_to_can_msg  a_rx_buf_to_can_msg (
    .can_rx_buf_in      (test_can_rx_buf),
    .can_msg_out        (test_can_msg),
    .is_ext_data_frame  ()
);


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

destination_filter  the_destination_filter (
   .reset_n      (reset_n),
   .ck           (ck),
   .stn_id       (stn_id),
   .rcv_data_in  (rcv_data_bus),
   .can_msg_out  (can_msg_bus)
);

assign  trig = can_msg_bus.trig;



//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    expected_msg = '0;
    actual_msg = '0;
    rcv_data_bus.rx_buf = '0; rcv_data_bus.trig = 1'b0;
    stn_id = DEST_STN_ID;
    actual_level = 1'b0; expected_level = 1'b0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------
    // (1) (全局宛)  message に対する動作を確認する。

    // test 用の受信メッセージの準備を行う。
    test_can_rx_buf = '0;  // SRR, RTR も 0 に設定される。
    {test_can_rx_buf.STD_ID, test_can_rx_buf.EXT_ID} = {
        1'b0, CONT_STN_ID, 4'b0, MSG_STATUS_REQUEST, ALL_STATION
    };
    test_can_rx_buf.IDE       = 1'b1;  // ext id で運用する。
    test_can_rx_buf.DLC[3:0]  = 4'd0;
    test_can_rx_buf.DATA_BYTE = 64'h0;

    rcv_data_bus.rx_buf = test_can_rx_buf;

    // 前段のモジュールからの trig を発生させる。
    @(posedge ck) rcv_data_bus.trig = 1'b1;
    @(posedge ck) rcv_data_bus.trig = 1'b0;

    // trig が出力されることを確認する。
    test.subtitle = "(1.1) trig wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(can_msg_bus.trig, timeout);

    // can_msg_bus に message がセットされる。
    test.subtitle = " (1.2) rcv message test.";
    expected_msg = test_can_msg;
    actual_msg   = can_msg_bus.msg;
    test.assert_equals(expected_msg, actual_msg, "hex");

    #(SIM_CLK_COUNT * 10);

    //-----------------------------------------------------------------
    // (2) (自局宛)  message に対する動作を確認する。

    // test 用の受信メッセージの準備を行う。
    test_can_rx_buf = '0;  // SRR, RTR も 0 に設定される。
    {test_can_rx_buf.STD_ID, test_can_rx_buf.EXT_ID} = {
        1'b0, CONT_STN_ID, 4'b0, MSG_CONFIG_DATA_WRITE, DEST_STN_ID
    };
    test_can_rx_buf.IDE       = 1'b1;  // ext id で運用する。
    test_can_rx_buf.DLC[3:0]  = 4'd8;
    test_can_rx_buf.DATA_BYTE = 64'h1122334455667788;

    rcv_data_bus.rx_buf = test_can_rx_buf;

    // 前段のモジュールからの trig を発生させる。
    @(posedge ck) rcv_data_bus.trig = 1'b1;
    @(posedge ck) rcv_data_bus.trig = 1'b0;

    // trig が出力されることを確認する。
    test.subtitle = "(2.1) trig wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(can_msg_bus.trig, timeout);

    // can_msg_bus に message がセットされる。
    test.subtitle = " (2.2) rcv message test.";
    expected_msg = test_can_msg;
    actual_msg   = can_msg_bus.msg;
    test.assert_equals(expected_msg, actual_msg, "hex");


    //-----------------------------------------------------------------
    // (3) (他局宛)  message に対する動作を確認する。

    // test 用の受信メッセージの準備を行う。
    test_can_rx_buf = '0;
    {test_can_rx_buf.STD_ID, test_can_rx_buf.EXT_ID} = {
        1'b0, CONT_STN_ID, 4'b0, MSG_CONFIG_DATA_WRITE, OTHER_STN_ID
    };
    test_can_rx_buf.IDE       = 1'b1;  // ext id で運用する。
    test_can_rx_buf.DLC[3:0]  = 4'd8;
    test_can_rx_buf.DATA_BYTE = 64'h1122334455667788;

    rcv_data_bus.rx_buf = test_can_rx_buf;

    // 前段のモジュールからの trig を発生させる。
    @(posedge ck) rcv_data_bus.trig = 1'b1;
    @(posedge ck) rcv_data_bus.trig = 1'b0;

    // trig が出力されないことを確認する。
    test.subtitle = "(3.1) trig is not occurr.";
    timeout = SIM_COUNT_1MS;
    test.assert_not_occurr(can_msg_bus.trig, timeout);






    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

