//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// tx buffer 2 を使用した送信処理の動作を確認する。
//   (1) 送信バッファへの書き込み動作を確認する。 
//   (2) 送信要求ビットの書き込み動作を確認する。 
//   (3) done が出力されることを確認する。
//
module   test107_03_msg_data_send_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test107_03_msg_data_send_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"

`include "MCP2515_types.svh"
`include "can_msg_types.svh"

`include "MCP2515.svh"
`include "can_msg.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  CONT_STN_ID = 8'h03;
localparam  DEST_STN_ID = 8'h05;

// 使用するコマンドの最大サイズ (=4)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;

logic   cont_start;
logic   cont_done;

logic   [2:0] txbuf_busy;
can_msg_if  can_msg_bus();

can_msg_t   test_can_msg;
can_tx_buf_t    expected_tx_buf;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

msg_data_send #(
    .TXBUF_NUM  (2)
) the_msg_data_send (
    .ck             (ck),
    .reset_n        (reset_n),
    .txbuf_busy     (txbuf_busy),
    .done           (cont_done),
    .can_msg_in     (can_msg_bus),
    .cont_out       (spi_cont_bus)
);

assign  can_msg_bus.msg = test_can_msg;
assign  can_msg_bus.trig    = cont_start;


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    cont_start = 1'b0;
    txbuf_busy = 3'b111;
    test_can_msg = '0;
    expected_tx_buf = '0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------
    // (1) 送信バッファへの書き込み動作を確認する。 

    // (1.1) start がかかる前は、 request は off 。
    test.subtitle = "(1.1) request is off before start.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");

    // テスト用の can_msg を設定する。
    test_can_msg = '0;
    test_can_msg.src_stn_id = CONT_STN_ID;
    test_can_msg.msg_id = MSG_CONFIG_DATA_WRITE;
    test_can_msg.dest_stn_id = DEST_STN_ID;
    test_can_msg.dlc  = 4'd8;
    test_can_msg.data = 64'h1122334455667788;

    // tx_buf の期待値を設定する。
    expected_tx_buf = '0;
    {
        expected_tx_buf.STD_ID, expected_tx_buf.EXT_ID
    } = {
        1'b0, CONT_STN_ID, 4'b0, MSG_CONFIG_DATA_WRITE, DEST_STN_ID
    };
    expected_tx_buf.EXIDE = 1'b1;
    expected_tx_buf.RTR   = 1'b0;
    expected_tx_buf.DLC   = 4'd8;
    expected_tx_buf.DATA_BYTE = 64'h1122334455667788;


    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = 14*8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // start をかける。
    @(posedge ck) cont_start = 1'b1;
    @(posedge ck) cont_start = 1'b0;

    // tx buf が busy の場合は、start が保留されることを確認する。
    test.subtitle = "(1.2) request is not occurr when txbuf busy.";
    timeout = SIM_COUNT_1US;
    test.assert_not_occurr(spi_cont_bus.request, timeout);

    // tx buf busy が off になったら request が出力されることを確認する。
    @(posedge ck) txbuf_busy = 3'b011;

    test.subtitle = "(1.3) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(1.4) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(1.5) load to txbuf 0 test.";
    expected_data = {CMD_LOAD_TX_BUF_AT_TXB2_SIDH, expected_tx_buf};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (2) 送信要求ビットの書き込み動作を確認する。 

    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = 1*8;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // SPI 通信の完了を待ち、受信したデータと期待値を照合する。
    test.subtitle = "(2.1) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(2.2) request to send 0 test.";
    expected_data = {CMD_REQ_TO_SEND_TXB2};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-----------------------------------------------------------------
    // (3) done が出力されることを確認する。

    // 完了後 done が出力される。
    test.subtitle = "(3) can_device_status_read cont_done wait.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(cont_done, timeout);




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

