//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) port index は、 1 ~ PORT_NUM とする。 (0 の場合は、出力 0 )
// (2) 優先順位は、port 1 が最も高い。
//
module  spi_cont_sel #(
    parameter PORT_NUM = 0
) (
    input   wire   reset_n,
    input   wire   ck,

    spi_cont_comb_if.in     cont_comb_in,
    spi_cont_if.out         cont_sel,

    output  logic   port_request,
    input   wire    port_permit
);

`include "assert_param.svh"
initial begin
    `assert_param(PORT_NUM > 0, PORT_NUM)
end


localparam  MSB = $clog2(PORT_NUM);

`include "spi_cont_comb.svh"

spi_cont_mo_t   [PORT_NUM:1] cont_in_mo;
spi_cont_so_t   [PORT_NUM:1] cont_in_so;
spi_cont_mo_t   cont_sel_mo;
spi_cont_so_t   cont_sel_so;

//-------------------------------------------------------------
logic   [MSB:0] index[PORT_NUM:0];
logic   [MSB:0] req_idx;
logic   [MSB:0] idx;

spi_cont_mo_t   [PORT_NUM:0] cont_in_mo_0;

//-------------------------------------------------------------
assign  index[0] = '0;

genvar i;
generate
for (i=1; i<=PORT_NUM; i++) begin : generate_assign_block

    // cont_comb_in, cont_in_mo/so 間を assign する。
    assign  cont_in_mo[i] = cont_comb_in.mo[i];
    assign  cont_comb_in.so[i] = cont_in_so[i];

    // index 決定ブロック。 最終的に index[PORT_NUM] が判定結果になる。
    port_index #(
        .PORT_NUM   (PORT_NUM),
        .INDEX      (i)
    ) 
    port_index_inst(
        .request    (cont_in_mo[i].request),
        .index_in   (index[i-1]),
        .index_out  (index[i])
    );

    // slave out 設定ブロック ---------  cont_sel_so.permit は未使用。
    assign  cont_in_so[i].permit      = (req_idx == i) ? port_permit : '0;

    assign  cont_in_so[i].done        = (idx == i) ? cont_sel_so.done        : '0;
    assign  cont_in_so[i].shift_carry = (idx == i) ? cont_sel_so.shift_carry : '0;
    assign  cont_in_so[i].miso        = (idx == i) ? cont_sel_so.miso        : '0;
end
endgenerate


// 処理中に 優先度が高い request が入っても動作に影響しないように
// port_permit が on の間は、 req_idx を更新しないようにする。
always_ff @(posedge ck or negedge reset_n) begin
    if ( ~reset_n )
        req_idx <= '0;
    else if ( ~port_permit )
        req_idx <= index[PORT_NUM];
    else
        req_idx <= req_idx;
end


// idx : PORT_NUM ~ 1 (有効), 0(0 固定出力) となるようにする。
assign  cont_in_mo_0 = {cont_in_mo, {$bits(cont_sel_mo){1'b0}}};

assign  port_request = cont_in_mo_0[req_idx].request;

assign  idx = ( port_permit ) ? req_idx : '0;

assign  cont_sel_mo  = cont_in_mo_0[idx];


// interface にまとめて cont_sel ポートに出力する。
spi_cont_mixer  a_spi_cont_mixer (
    .cont_mo    (cont_sel_mo),
    .cont_so    (cont_sel_so),
    .cont_out   (cont_sel)
);




endmodule


//-------------------------------------------------------------
module  port_index #(
    parameter PORT_NUM = 0,
    parameter INDEX = 0
) (
    request, index_in, index_out
);

localparam  MSB = $clog2(PORT_NUM);

input   wire    request;
input   wire    [MSB:0] index_in;
output  logic   [MSB:0] index_out;

//-------------------------------------------------------------

always_comb begin
    if (index_in != 0) 
        index_out = index_in;
    else if (request == 1'b1)
        index_out = INDEX[MSB:0];
    else
        index_out = '0;
end



endmodule

