//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

module  s1frm0001_core (
    input   wire    CLK,

    output  wire    CAN_CSn,
    input   wire    CAN_INTn,
    output  wire    CAN_SCK,
    output  wire    CAN_SI,
    input   wire    CAN_SO,

    output  wire    FRAM_CSn,
    output  wire    FRAM_SCK,
    output  wire    FRAM_SI,
    input   wire    FRAM_SO,

    //output  wire    LED1, // CONF_DONE に接続している。
    output  wire    LED2n,

    //output  wire    LVDS_REn,
    //output  wire    [7:0]  LVDS_D,
    //output  wire    [7:0]  LVDS_DEn,
    //input   wire    [7:0]  LVDS_R,

    input   wire    [3:0]  SW1,
    input   wire    [3:0]  SW2,

    input   wire    [31:0] DI
);

`include "clock.svh"
`include "can_device_status.svh"
`include "can_msg.svh"

localparam  CONFIG_DATA_INIT_READ_DELAY_MS = 7'd100;
localparam  LED2_INTERVAL_MS    = 11'd2000;
localparam  LED2_PULSE_WIDTH_MS = 11'd40;
localparam  SPI_CONT_PORT_NUM = 5;
localparam  CONFIG_DATA_BYTE_LEN = 8;
localparam  DI_BYTE_LEN  = 4;
localparam  DI_BIT_WIDTH = DI_BYTE_LEN * 8;
localparam  HALF_OF_CAN_SCK_PRI = 2;    // 10M bps


//-----------------------------------------------------------
wire    ck;
wire    reset_n;
wire    carry_1ms;

wire    led2_on;

wire    [7:0] hex_sw;
wire    [7:0] stn_id;

spi_cont_if     can_dev_sts_read_cont();
spi_cont_if     can_device_config_cont();
spi_cont_if     config_data_msg_tx_cont();
spi_cont_if     di_data_status_msg_tx_cont();
spi_cont_if     can_rcv_msg_read_cont();

spi_cont_comb_if # (
    .PORT_NUM   (SPI_CONT_PORT_NUM)
) 
spi_cont_comb();

spi_if  can_spi_bus();
spi_if  fram_spi_bus();

spi_cont_if     spi_cont_bus();

rcv_data_if     rcv_data_out();
can_msg_if      rcv_can_msg_bus();

can_dev_sts_t   can_dev_sts;

wire    can_int;
wire    config_data_init_read;
wire    config_data_init_read_done;
wire    config_data_write;
wire    config_data_write_done;
wire    [CONFIG_DATA_BYTE_LEN*8-1:0] rcv_config_data;
wire    [CONFIG_DATA_BYTE_LEN*8-1:0] config_data;

wire    can_device_config_done;
wire    can_device_en;

wire    status_request_trig;
wire    config_data_request_trig;
wire    [2:0] txbuf_busy;

wire    [DI_BIT_WIDTH-1:0] di_data;
wire    di_changed;
wire    [7:0] di_duration;

wire    dev_sts_update;

//-----------------------------------------------------------
assign  ck = CLK;   // S1MOD0001 では、40MHz


reset_gen   internal_reset (
    .ck         (ck),
    .reset_n    (reset_n) 
);


carry_gen #(
    .INTERVAL   (CK_COUNT_1MS)
) 
carry_gen_1ms (
    .*,
    .en         (1'b1),
    .carry      (carry_1ms)
);


pulse_gen #(
    .CNT_BIT_WIDTH  ($bits(LED2_INTERVAL_MS))
) 
LED2_blinker (
    .*,
    .carry          (carry_1ms),
    .interval       (LED2_INTERVAL_MS),
    .pulse_width    (LED2_PULSE_WIDTH_MS),
    .en             (1'b1),
    .out            (led2_on)
);

assign  LED2n = ~led2_on;


assign  hex_sw = ~{SW1, SW2};
assign  stn_id = hex_sw;


//-----------------------------------------------------------
spi_cont_combiner   can_spi_cont_combiner ( 
    .port_1     (can_dev_sts_read_cont),
    .port_2     (can_device_config_cont),
    .port_3     (config_data_msg_tx_cont),
    .port_4     (di_data_status_msg_tx_cont),
    .port_5     (can_rcv_msg_read_cont),
    .comb_out   (spi_cont_comb)
);

spi_cont_switch  #(
    .PORT_NUM   (SPI_CONT_PORT_NUM)
) 
can_spi_cont_switch (
    .*,
    .interrupt      (can_int),
    .dev_sts_update (dev_sts_update),
    .cont_comb_in   (spi_cont_comb),
    .cont_out       (spi_cont_bus)
);

spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_CAN_SCK_PRI)
)
can_spi_cont (
    .*,
    .cont_in    (spi_cont_bus),
    .spi_port   (can_spi_bus)
); 

assign  CAN_CSn = ~can_spi_bus.cs;
assign  CAN_SCK = can_spi_bus.sck;
assign  CAN_SI  = can_spi_bus.mosi;
assign  can_spi_bus.miso = CAN_SO;
assign  can_int = ~CAN_INTn; 


//-----------------------------------------------------------
one_shot_trig #(
    .CNT_BIT_WIDTH   ($bits(CONFIG_DATA_INIT_READ_DELAY_MS))
) 
config_data_init_read_trig_gen (
    .*,
    .carry      (carry_1ms),
    .delay      (CONFIG_DATA_INIT_READ_DELAY_MS),
    .en         (1'b1),
    .out        (config_data_init_read)
);

config_data_manager # (
    .BYTE_LEN   (CONFIG_DATA_BYTE_LEN)
)
conf_data_mangr (
    .*,
    .read_cmd   (config_data_init_read),
    .read_done  (config_data_init_read_done),
    .write_cmd  (config_data_write),
    .write_done (config_data_write_done),
    .din        (rcv_config_data),
    .dout       (config_data),
    .spi_port   (fram_spi_bus)
);

assign  FRAM_CSn = ~fram_spi_bus.cs;
assign  FRAM_SCK = fram_spi_bus.sck;
assign  FRAM_SI  = fram_spi_bus.mosi;
assign  fram_spi_bus.miso = FRAM_SO;


//-----------------------------------------------------------
can_device_status_read     can_dev_sts_read (
    .*,
    .start          (dev_sts_update),
    .done           (),
    .can_dev_sts    (can_dev_sts),
    .txbuf_busy     (txbuf_busy),
    .cont_out       (can_dev_sts_read_cont)
);


can_device_config   can_dev_config (
    .*,
    .carry_1ms  (carry_1ms),
    .start      (config_data_init_read_done),
    .cont_out   (can_device_config_cont),
    .done       (can_device_config_done)
);

sr_latch   can_device_config_done_latch (
    .* ,
    .set    (can_device_config_done),
    .rst    (1'b0),
    .out    (can_device_en)
);


config_data_msg #(
    .BYTE_LEN  (CONFIG_DATA_BYTE_LEN)
) 
conf_data_msg (
    .*,
    .trig           (config_data_request_trig),
    .txbuf_busy     (txbuf_busy),
    .stn_id         (stn_id),
    .config_data    (config_data),
    .cont_out       (config_data_msg_tx_cont)
);


//-----------------------------------------------------------
can_rcv_msg_read    rcv_msg_read (
    .*,

    .stn_id         (stn_id),
    .can_dev_sts    (can_dev_sts),

    .rcv_data_out   (rcv_data_out),
    .cont_out       (can_rcv_msg_read_cont)
);

destination_filter  dest_filter (
    .*,
   .stn_id       (stn_id),
   .rcv_data_in  (rcv_data_out),
   .can_msg_out  (rcv_can_msg_bus)
);


//-----------------------------------------------------------
request_msg #(
    .MSG_ID     (MSG_STATUS_REQUEST)
)  
sts_req_msg (
    .*,
    .can_msg_in (rcv_can_msg_bus),
    .trig_out   (status_request_trig)
);


request_msg #(
    .MSG_ID     (MSG_CONFIG_DATA_REQUEST)
)  
conf_data_req_msg (
    .*,
    .can_msg_in (rcv_can_msg_bus),
    .trig_out   (config_data_request_trig)
);


parallel_out_msg #(
    .MSG_ID     (MSG_CONFIG_DATA_WRITE),
    .DLC_MIN    (CONFIG_DATA_BYTE_LEN),
    .DLC_MAX    (CONFIG_DATA_BYTE_LEN),
    .BIT_WIDTH  (CONFIG_DATA_BYTE_LEN*8)
)  
conf_data_write_msg (
    .*,
    .can_msg_in (rcv_can_msg_bus),
    .data       (rcv_config_data),
    .trig_out   (config_data_write)
);



//-----------------------------------------------------------
di_data_status_msg #(
    .BYTE_LEN  (DI_BYTE_LEN)
) 
di_data_sts_msg (
    .*,
    .trig           ((di_changed & can_device_en) | status_request_trig),
    .txbuf_busy     (txbuf_busy),
    .stn_id         (stn_id),
    .di_data        (di_data),
    .cont_out       (di_data_status_msg_tx_cont)
);



digital_input #(
    .SIG_BIT_WIDTH      ($bits(DI)),
    .CNT_BIT_WIDTH      ($bits(di_duration))
) 
din_port (
    .*,
    .carry      (carry_1ms),
    .duration   (di_duration),

    .data_in    (DI),
    .data_out   (di_data),
    .changed    (di_changed)
);

assign  di_duration = config_data[7:0];




endmodule

