//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) en から 最初の出力が行われるまでの時間を計測する。
// (2) 出力信号の１周期の時間を計測する。
// (3) 出力信号の ON 時間を計測する。
//
module   test002_01_interval_timer_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test002_01_interval_timer_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;

time    actual;
time    expected;

logic   carry_gen_en;
logic   carry_1ms;

logic   timer_en;
logic   timer_out;


// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
localparam INTERVAL_1MS = 40000;    // pri = ck * 4000 (1ms 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (INTERVAL_1MS)
) 
a_carry_gen (
    .ck         (ck),
    .reset_n    (reset_n),
    .en         (carry_gen_en),
    .carry      (carry_1ms)
);


//-----------------------------------------------------------
//  テスト対象のインスタンス
localparam TIMER_INTERVAL_MS = 4'd10;

interval_timer #(
    .CNT_BIT_WIDTH  ($bits(TIMER_INTERVAL_MS))
) 
the_interval_timer (
    .ck         (ck),
    .reset_n    (reset_n),
    .carry      (carry_1ms),
    .interval   (TIMER_INTERVAL_MS),
    .en         (timer_en),
    .out        (timer_out)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

time    exp_lower, exp_upper;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;

    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    timer_en = 1'b0;

    #100  reset_n = 1'b1;
    #500;

    //-------------------------------------------------------------
    // (1) en から 最初の出力が行われるまでの時間を計測する。
    test.subtitle = "(1) measure the time from timer_en to timer out.";
    expected = TIMER_INTERVAL_MS * SIM_COUNT_1MS; 
    exp_lower = expected - SIM_COUNT_1MS;
    exp_upper = expected + SIM_COUNT_1MS;
    timeout = expected + SIM_COUNT_1MS;


    @(posedge ck);
    #1;
    carry_gen_en = 1'b1; timer_en = 1'b1; // 次の ck で有効になる。 

    @(posedge ck); 
    tim1.start();
    test.assert_pos_edge(timer_out, timeout);

    actual = tim1.lap_time();
    test.assert_range(exp_lower, exp_upper, actual, "time");


    //-------------------------------------------------------------
    // (2) 出力信号の１周期の時間を計測する。
    test.subtitle = "(2.1) interval time test. : timer_out wait";
    expected = TIMER_INTERVAL_MS * SIM_COUNT_1MS; 

    test.assert_pos_edge(timer_out, timeout);

    actual = tim1.lap_time();
    test.subtitle = "(2.2) interval time test.";
    test.assert_equals(expected, actual, "time");

    @(negedge timer_out);


    //-------------------------------------------------------------
    // (3) 出力信号の ON 時間を計測する。
    test.subtitle = "(3) timer_out time width test.";
    expected = SIM_CLK_COUNT; 

    @(posedge timer_out);
    tim1.start();

    @(negedge timer_out);
    actual = tim1.lap_time();
    test.assert_equals(expected, actual, "time");



    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end



endmodule

