//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) timer_out の検出時間を計測する。
// (2) インクリメントして出力が off になっていないか？
//
module   test003_01_on_timer_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test003_01_on_timer_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;

time    actual;
time    expected;

logic   carry_gen_en;
logic   carry_1ms;

logic   timer_in;
logic   timer_out;

// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
localparam CARRY_INTERVAL = 40000;  // pri = ck * 4000 (1ms 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (CARRY_INTERVAL)
) 
a_carry_gen (
    .ck         (ck),
    .reset_n    (reset_n),
    .en         (carry_gen_en),
    .carry      (carry_1ms)
);



//-----------------------------------------------------------
//  テスト対象のインスタンス
localparam DELAY = 3'd5;

on_timer #(
    .CNT_BIT_WIDTH  ($bits(DELAY))
) 
the_on_timer (
    .ck         (ck),
    .reset_n    (reset_n),
    .carry      (carry_1ms),
    .delay      (DELAY),
    .in         (timer_in),
    .out        (timer_out)
);



//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

time    exp_lower, exp_upper;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;
    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    timer_in = 1'b0;
    actual = 0; expected = 0;
    exp_lower = 0; exp_upper = 0;
    timeout = 0;


    #100 reset_n = 1'b1;
    #200 carry_gen_en = 1'b1;

    //-------------------------------------------------------------
    // (1) timer_out の検出時間を計測する。
    expected = (CARRY_INTERVAL * SIM_CLK_COUNT) * DELAY;

    @(negedge carry_1ms);
    tim1.start();
    timer_in = 1'b1;

    test.subtitle = "(1.1) on_timer detect time test.(posedge timer_out)";
    timeout = expected * 100;
    test.assert_pos_edge(timer_out, timeout);

    actual = tim1.lap_time();
    test.subtitle = "(1.2) on_timer detect time test.";
    test.assert_equals(expected, actual, "time");


    //-------------------------------------------------------------
    // (2) インクリメントして出力が off になっていないか？
    @(posedge carry_1ms);
    @(posedge carry_1ms);

    test.subtitle = "(2) timer_out is 1'b1 after detect.";
    test.assert_equals(1'b1, timer_out, "bin");



    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end



endmodule

