//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) en から trig の出力が行われるまでの時間を計測する。
//
module   test004_01_one_shot_trig_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test004_01_one_shot_trig_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;

time    actual;
time    expected;

logic   carry_gen_en;
logic   carry_1ms;

logic   enable;
logic   trig_out;

// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
localparam CARRY_INTERVAL = 40000;  // pri = ck * 4000 (1ms 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (CARRY_INTERVAL)
) 
a_carry_gen (
    .ck         (ck),
    .reset_n    (reset_n),
    .en         (carry_gen_en),
    .carry      (carry_1ms)
);

// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス
localparam TRIG_DELAY_MS = 5'd20;

one_shot_trig #(
    .CNT_BIT_WIDTH   ($bits(TRIG_DELAY_MS))
) 
the_one_shot_trig (
    .ck         (ck),
    .reset_n    (reset_n),
    .carry      (carry_1ms),
    .delay      (TRIG_DELAY_MS),
    .en         (enable),
    .out        (trig_out)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

time    exp_lower, exp_upper;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;
    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    enable = 1'b0;
    actual = 0; expected = 0;
    exp_lower = 0; exp_upper = 0;
    timeout = 0;


    #100 reset_n = 1'b1;
    #200 carry_gen_en = 1'b1;


    //-------------------------------------------------------------
    // (1) en から trig の出力が行われるまでの時間を計測する。
    test.subtitle = "(1.1) measure the time from enable to trig out.";

    // on_timer と rise_detector を組合わせているので 1CLK 多めになる。
    expected = (TRIG_DELAY_MS * SIM_COUNT_1MS) + SIM_CLK_COUNT; 

    test.subtitle = "(1.1) measure the time from enable to trig out. : carry_1ms wait.";
    @(negedge carry_1ms); // negedge
    enable = 1'b1;
    tim1.start();

    test.subtitle = "(1.2) measure the time from enable to trig out. : trig_out wait.";
    timeout = expected + SIM_COUNT_1MS;
    test.assert_pos_edge(trig_out, timeout);

    actual = tim1.lap_time();
    test.subtitle = "(1.3) measure the time from enable to trig out.";
    test.assert_equals(expected, actual, "time");




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end



endmodule

