//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) 出力信号の ON 時間を計測する。
// (2) 出力信号の１周期の時間を計測する。
//
module   test010_01_pulse_gen_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test010_01_pulse_gen_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;

logic   carry_gen_en;
logic   carry;

time    expected;
time    actual;

logic   pg_en;
logic   pg_out;


// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
localparam CARRY_INTERVAL = 40000;  // pri = ck * 4000 (1ms 相当)

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (CARRY_INTERVAL)
) 
a_carry_gen (
    .ck         (ck),
    .reset_n    (reset_n),
    .en         (carry_gen_en),
    .carry      (carry)
);


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス
localparam INTERVAL    = 4'd10;
localparam PULSE_WIDTH = 4'd3;

pulse_gen #(
    .CNT_BIT_WIDTH  ($bits(INTERVAL))
) 
the_pulse_gen (
    .ck             (ck),
    .reset_n        (reset_n),
    .carry          (carry),
    .interval       (INTERVAL),
    .pulse_width    (PULSE_WIDTH),
    .en             (pg_en),
    .out            (pg_out)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;

    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    pg_en = 1'b0;
    actual = 0; expected = 0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    carry_gen_en = 1'b1;

    @(posedge carry);

    pg_en = 1'b1;

    timeout = (INTERVAL * SIM_COUNT_1MS) + SIM_COUNT_1MS;

    //-------------------------------------------------------------
    // (1) 出力信号の ON 時間を計測する。
    test.subtitle = "(1.1) pulse on time test. : pulse on wait";
    test.assert_pos_edge(pg_out, timeout);
    tim1.start();

    test.subtitle = "(1.2) pulse on time test. : pulse off wait";
    test.assert_neg_edge(pg_out, timeout);
   

    test.subtitle = "(1.3) pulse on time test.";
    actual = tim1.lap_time();

    expected = PULSE_WIDTH * SIM_COUNT_1MS; 

    test.assert_equals(expected, actual, "time");


    //-------------------------------------------------------------
    // (2) 出力信号の１周期の時間を計測する。
    test.subtitle = "(2.1) pulse interval time test. : pulse on wait";
    test.assert_pos_edge(pg_out, timeout);

    test.subtitle = "(2.2) pulse interval time test.";
    expected = (INTERVAL * SIM_COUNT_1MS); 
    actual = tim1.split_time();
    test.assert_equals(expected, actual, "time");




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

