//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// CS の動作を確認する。(内部では正論理)
//   (1) start するまでは、L 状態を保持する。 
//   (2) start 後に H に遷移する。
//   (3) バイト数に応じて H が継続して、その後 L になる。
//   (4) sck 1クロック経過後 done が出力される。
//
module   test102_04_spi_cont_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test102_04_spi_cont_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;
time    expected_time;
time    actual_time;
logic   expected_level;
logic   actual_level;

// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 
localparam   HALF_OF_SCK_PRI = 10;
localparam   SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:2MHz 
localparam   BYTE_LEN = 4;

logic   sck;
logic   cs;
spi_cont_if     spi_cont_bus();
spi_if          spi_bus();
                             
spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
the_spi_cont (
    .ck         (ck),
    .reset_n    (reset_n),
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    spi_cont_bus.start = 1'b0;
    spi_cont_bus.request = 1'b0;
    spi_cont_bus.byte_num = BYTE_LEN;
    actual_time = 0; expected_time = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    timeout = 0;


    #100  reset_n = 1'b1;

    @(posedge ck);

    //-------------------------------------------------------------
    //  (1) start するまでは、L 状態を保持する。 
    test.subtitle = "(1) cs is 1'b0 until start.";
    expected_level = 1'b0;
    actual_level = spi_bus.cs;
    test.assert_equals(expected_level, actual_level, "bin");


    //-------------------------------------------------------------
    //  (2) start 後に H に遷移する。
    test.subtitle = "(2.1) cs is 1'b1 just after start. : wait for cs rise.";
    timeout = SIM_COUNT_1US; 
    @(posedge ck) spi_cont_bus.start = 1'b1;
    @(posedge ck) spi_cont_bus.start = 1'b0;
    tim1.start();

    test.assert_pos_edge(cs, timeout);


    //-------------------------------------------------------------
    //  (3) バイト数に応じて H が継続して、その後 L になる。
    test.subtitle = "(3.1) cs assert time test. : wait for cs fall.";
    expected_time = BYTE_LEN * 8 * SCK_PRI * SIM_CLK_COUNT;
    timeout = expected_time + SIM_COUNT_1US;

    test.assert_neg_edge(cs, timeout);

    test.subtitle = "(3.2) cs assert time test.";
    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");


    //-------------------------------------------------------------
    //  (4) sck 1クロック経過後 done が出力される。
    test.subtitle = "(4.1) done output timing test. : wait for done rise.";
    expected_time = SCK_PRI * SIM_CLK_COUNT;
    timeout = expected_time + SIM_COUNT_1US;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    test.subtitle = "(4.2) done output timing test.";
    actual_time = tim1.lap_time();
    test.assert_equals(expected_time, actual_time, "time");





    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

