//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// SPI bus を通して コマンド出力と、レスポンス入力を確認する。
//  (1) 出力動作以外の状態では、mosi が Hi 状態であることを確認する。 
//  (2) 処理を開始して終了(done)を待つ。
//  (3) shift_carry, sck の回数を照合する。
//  (4) コマンドを照合する。
//  (5) レスポンスを照合する。
//
module   test102_05_spi_cont_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test102_05_spi_cont_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;
time    timeout;
time    expected_time;
time    actual_time;
logic   expected_level;
logic   actual_level;
integer expected_count;
integer actual_count;



// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"


logic   sck;
logic   cs;
spi_cont_if     spi_cont_bus();
spi_if          spi_bus();

logic   trig, done;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


localparam  SPI_COMMAND   = 24'hABCDEF;
localparam  SPI_RESPONSE  = 40'h8234567890;
localparam  SREG_BIT_LEN  = $bits(SPI_COMMAND) + $bits(SPI_RESPONSE);
localparam  SREG_BYTE_LEN = SREG_BIT_LEN / 8;

logic   [(SREG_BIT_LEN-1):0] expected_data;
logic   [(SREG_BIT_LEN-1):0] actual_data;


sim_spi_cmd #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_cmd (
   .ck        (ck),
   .trig      (trig),
   .done      (done),
   .spi_cont  (spi_cont_bus)
);


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 
localparam   HALF_OF_SCK_PRI = 2;
localparam   SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 


spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
the_spi_cont (
    .ck         (ck),
    .reset_n    (reset_n),
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;

integer     shift_carry_count;
integer     sck_count;

initial begin
    shift_carry_count = 0;

    forever begin
        @(posedge spi_cont_bus.shift_carry);
        shift_carry_count++;
    end

end


initial begin
    sck_count = 0;

    forever begin
        @(posedge spi_bus.sck);
        sck_count++;
    end

end


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    trig = 1'b0;
    actual_time = 0; expected_time = 0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_count = 0; expected_count = 0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #100;

    @(posedge ck);

    //-------------------------------------------------------------
    //  (1) 出力動作以外の状態では、mosi が Hi 状態であることを確認する。 
    test.subtitle = "(1) mosi is hi when input mode.";
    expected_level = 1'b1;
    actual_level = spi_bus.mosi;
    test.assert_equals(expected_level, actual_level, "bin");


    //-------------------------------------------------------------
    //  (2) 処理を開始して終了(done)を待つ。

    // spi slave にテストデータを設定する。
    a_sim_spi_slave.cmd_bit_len = $bits(SPI_COMMAND);
    a_sim_spi_slave.res_bit_len = $bits(SPI_RESPONSE);
    a_sim_spi_slave.response    = SPI_RESPONSE;

    a_sim_spi_cmd.cmd_bit_len = $bits(SPI_COMMAND);
    a_sim_spi_cmd.res_bit_len = $bits(SPI_RESPONSE);
    a_sim_spi_cmd.command     = SPI_COMMAND;


    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    test.subtitle = "(2) wait for done";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(done, timeout);
    
    @(negedge done);
    
    //-------------------------------------------------------------
    //  (3) shift_carry, sck の回数を照合する。
    test.subtitle = "(3.1) shift_carry_count test.";
    expected_count = SREG_BIT_LEN;
    actual_count   = shift_carry_count;
    test.assert_equals(expected_count, actual_count, "int");

    test.subtitle = "(3.2) sck_count test.";
    expected_count = SREG_BIT_LEN;
    actual_count   = sck_count;
    test.assert_equals(expected_count, actual_count, "int");


    //-------------------------------------------------------------
    //  (4) コマンドを照合する。
    test.subtitle = "(4) command data test.";
    expected_data = SPI_COMMAND;
    actual_data   = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    //-------------------------------------------------------------
    //  (5) レスポンスを照合する。
    test.subtitle = "(5) response data test.";
    expected_data = SPI_RESPONSE;
    actual_data   = a_sim_spi_cmd.rcv_response;
    test.assert_equals(expected_data, actual_data, "hex");





    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

