//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

// メッセージの受信動作を確認する。
// (1) 対象外の msg id のメッセージが入力された場合の動作を確認する。
// (2) 範囲外の dlc    のメッセージが入力された場合の動作を確認する。
// (3) 対象とするメッセージが入力された場合の動作を確認する。
// (4) DLC が 0 のメッセージを受信できるか確認する。
//
module   test108_01_msg_data_recv_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test108_01_msg_data_recv_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

`include "can_msg_types.svh"
`include "can_msg.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  CONT_STN_ID = 8'h03;
localparam  DEST_STN_ID = 8'h05;


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;

logic   trig;


can_msg_if  rcv_can_msg_bus();
can_msg_if  config_data_write_msg_out();
can_msg_if  status_request_msg_out();

can_msg_t   test_can_msg;
can_msg_t   expected_msg, actual_msg;
can_msg_t   config_data_write_msg;
can_msg_t   status_request_msg;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

// config data write
msg_data_recv #(
    .MSG_ID     (MSG_CONFIG_DATA_WRITE),
    .DLC_MIN    (8),
    .DLC_MAX    (8)
)  
the_msg_data_recv (
    .reset_n        (reset_n),
    .ck             (ck),
    .can_msg_in     (rcv_can_msg_bus),
    .can_msg_out    (config_data_write_msg_out)
);

// status request
msg_data_recv #(
    .MSG_ID     (MSG_STATUS_REQUEST),
    .DLC_MIN    (0),
    .DLC_MAX    (0)
)  
a_status_request_msg (
    .reset_n        (reset_n),
    .ck             (ck),
    .can_msg_in     (rcv_can_msg_bus),
    .can_msg_out    (status_request_msg_out)
);



assign  rcv_can_msg_bus.msg  = test_can_msg;
assign  rcv_can_msg_bus.trig = trig;


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    trig = 1'b0;
    test_can_msg = '0;
    actual_msg = 0; expected_msg = 0;
    config_data_write_msg = '0;
    status_request_msg = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    // テスト用の can_msg を設定する。
    config_data_write_msg = '0;
    config_data_write_msg.src_stn_id = CONT_STN_ID;
    config_data_write_msg.msg_id = MSG_CONFIG_DATA_WRITE;
    config_data_write_msg.dest_stn_id = DEST_STN_ID;
    config_data_write_msg.dlc  = 4'd8;
    config_data_write_msg.data = 64'h1122334455667788;

    status_request_msg = '0;
    status_request_msg.src_stn_id = CONT_STN_ID;
    status_request_msg.msg_id = MSG_STATUS_REQUEST;
    status_request_msg.dest_stn_id = DEST_STN_ID;
    status_request_msg.dlc  = 4'd0;
    status_request_msg.data = 64'h0;


    //-----------------------------------------------------------------
    // (1) 対象外の msg id のメッセージが入力された場合の動作を確認する。

    // 対象外のメッセージを設定する。
    test_can_msg = config_data_write_msg;
    test_can_msg.msg_id = MSG_DO_DATA_SET;

    // trig をかける。
    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    // (1.1) config data write が応答しないことを確認する。
    test.subtitle = "(1.1) config data write not detect.";
    timeout = SIM_COUNT_1US;
    test.assert_not_occurr(config_data_write_msg_out.trig, timeout);

    #(SIM_CLK_COUNT * 100); 

    // trig をかける。
    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    // (1.2) status request が応答しないことを確認する。
    test.subtitle = "(1.2) status request not detect.";
    timeout = SIM_COUNT_1US;
    test.assert_not_occurr(status_request_msg_out.trig, timeout);

    //-----------------------------------------------------------------
    // (2) 範囲外の dlc のメッセージが入力された場合の動作を確認する。
    test_can_msg = config_data_write_msg;
    test_can_msg.dlc = 4'd2;

    #(SIM_CLK_COUNT * 100); 

    // trig をかける。
    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    // (2.1) config data write が応答しないことを確認する。
    test.subtitle = "(2.1) config data write not detect.";
    timeout = SIM_COUNT_1US;
    test.assert_not_occurr(config_data_write_msg_out.trig, timeout);


    //-----------------------------------------------------------------
    // (3) config data write メッセージが入力された場合の動作を確認する。
    test_can_msg = config_data_write_msg;

    #(SIM_CLK_COUNT * 100); 

    // trig をかける。
    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    // (3.1) config data write が応答することを確認する。
    test.subtitle = "(3.1) config data write detect.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(config_data_write_msg_out.trig, timeout);

    // (3.2) message の内容を確認する。
    test.subtitle = "(3.2) config data write data test.";
    expected_msg = test_can_msg;
    actual_msg = config_data_write_msg_out.msg;
    test.assert_equals(expected_msg, actual_msg, "hex");



    //-----------------------------------------------------------------
    // (4) status request (DLC=0) のメッセージ受信を確認する。
    test_can_msg = status_request_msg;

    #(SIM_CLK_COUNT * 100); 

    // trig をかける。
    @(posedge ck) trig = 1'b1;
    @(posedge ck) trig = 1'b0;

    // (4.1) config data write が応答することを確認する。
    test.subtitle = "(4.1) config data write detect.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(status_request_msg_out.trig, timeout);

    // (4.2) message の内容を確認する。
    test.subtitle = "(4.2) config data write data test.";
    expected_msg = test_can_msg;
    actual_msg = status_request_msg_out.msg;
    test.assert_equals( expected_msg, actual_msg, "hex");






    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

