//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) 全ての port の request を on にして、動作を確認する。
// (2) 優先順位１位の request を off にして、動作を確認する。
// (3) 優先順位２位の request を off にして、動作を確認する。
//
module   test111_01_spi_cont_sel_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test111_01_spi_cont_sel_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

`include "spi_cont_comb.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定
localparam  PORT_NUM = 3;
localparam  MSB = $clog2(PORT_NUM);


//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
logic   [63:0] expected_data, actual_data;

spi_cont_comb_if # (
    .PORT_NUM   (PORT_NUM)
)
cont_comb_in();

spi_cont_if     cont_sel();

spi_cont_mo_t   cont_sel_mo;
spi_cont_so_t   cont_sel_so;

logic   port_request, port_permit;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


spi_cont_splitter  port_1_splitter (
    .cont_in    (cont_sel),
    .cont_mo    (cont_sel_mo),
    .cont_so    (cont_sel_so)
);


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

spi_cont_sel #(
    .PORT_NUM   (PORT_NUM)
)
the_spi_cont_sel (
    .reset_n            (reset_n),
    .ck                 (ck),
    .cont_comb_in       (cont_comb_in),
    .cont_sel           (cont_sel),
    .port_request       (port_request),
    .port_permit        (port_permit)
);



//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    cont_comb_in.mo[1] = '0;
    cont_comb_in.mo[2] = '0;
    cont_comb_in.mo[3] = '0;
    cont_sel_so = 4'b0;
    port_permit = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);


    cont_comb_in.mo[1].request  = 1'b0;
    cont_comb_in.mo[1].byte_num = 8'd11;
    cont_comb_in.mo[1].start    = 1'b1;
    cont_comb_in.mo[1].mosi     = 1'b1;

    cont_comb_in.mo[2].request  = 1'b0;
    cont_comb_in.mo[2].byte_num = 8'd22;
    cont_comb_in.mo[2].start    = 1'b1;
    cont_comb_in.mo[2].mosi     = 1'b1;

    cont_comb_in.mo[3].request  = 1'b0;
    cont_comb_in.mo[3].byte_num = 8'd33;
    cont_comb_in.mo[3].start    = 1'b1;
    cont_comb_in.mo[3].mosi     = 1'b1;


    //-----------------------------------------------------------------
    // (1) 全ての port の request を on にして、動作を確認する。
    cont_comb_in.mo[1].request  = 1'b1;
    cont_comb_in.mo[2].request  = 1'b1;
    cont_comb_in.mo[3].request  = 1'b1;

    @(posedge ck);

    // port request が出力される。
    test.subtitle = "(1.1) port_request wait.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(port_request, timeout);

    @(posedge ck);

    // この時点では、なにも選択されていない。
    test.subtitle = "(1.2) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    // permission 後、 port 1 が選択されている。
    port_permit = 1'b1;
    @(posedge ck);

    test.subtitle = "(1.3) port 1 select. : output";
    expected_data = cont_comb_in.mo[1];
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(1.4) port 1 select. : input";
    cont_sel_so = 4'b1010;
    expected_data = cont_sel_so;
    @(posedge ck);

    actual_data   = cont_comb_in.so[1];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(1.5) port 1 select. : port 2 nothing";
    expected_data = '0;
    actual_data   = cont_comb_in.so[2];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(1.6) port 1 select. : port 3 nothing";
    expected_data = '0;
    actual_data   = cont_comb_in.so[3];
    test.assert_equals(expected_data, actual_data, "bin");

    @(posedge ck);

    // permission を off にする。
    port_permit = 1'b0;

    @(posedge ck);

    // なにも選択されていない。
    test.subtitle = "(1.7) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    cont_comb_in.mo[1].request  = 1'b0;
    cont_comb_in.mo[2].request  = 1'b0;
    cont_comb_in.mo[3].request  = 1'b0;
    @(posedge ck);
    @(posedge ck);

    //-----------------------------------------------------------------
    // (2) 優先順位１位の request を off にして、動作を確認する。
    cont_comb_in.mo[2].request  = 1'b1;
    cont_comb_in.mo[3].request  = 1'b1;

    @(posedge ck);

    // port request が出力される。
    test.subtitle = "(2.1) port_request wait.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(port_request, timeout);

    @(posedge ck);

    // この時点では、なにも選択されていない。
    test.subtitle = "(2.2) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    // permission 後、 port 2 が選択されている。
    port_permit = 1'b1;
    @(posedge ck);

    test.subtitle = "(2.3) port 1 select. : output";
    expected_data = cont_comb_in.mo[2];
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(2.4) port 1 select. : input";
    cont_sel_so = 4'b1010;
    @(posedge ck);

    expected_data = '0;
    actual_data   = cont_comb_in.so[1];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(2.5) port 1 select. : port 2 nothing";
    expected_data = cont_sel_so;
    actual_data   = cont_comb_in.so[2];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(2.6) port 1 select. : port 3 nothing";
    expected_data = '0;
    actual_data   = cont_comb_in.so[3];
    test.assert_equals(expected_data, actual_data, "bin");

    @(posedge ck);

    // permission を off にする。
    port_permit = 1'b0;

    @(posedge ck);

    // なにも選択されていない。
    test.subtitle = "(2.7) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    cont_comb_in.mo[1].request  = 1'b0;
    cont_comb_in.mo[2].request  = 1'b0;
    cont_comb_in.mo[3].request  = 1'b0;
    @(posedge ck);


    //-----------------------------------------------------------------
    // (3) 優先順位２位の request を off にして、動作を確認する。
    cont_comb_in.mo[3].request  = 1'b1;

    @(posedge ck);

    // port request が出力される。
    test.subtitle = "(3.1) port_request wait.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(port_request, timeout);

    @(posedge ck);

    // この時点では、なにも選択されていない。
    //dspl_value("idx", the_spi_cont_sel.idx, "int");
    test.subtitle = "(3.2) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    // permission 後、 port 2 が選択されている。
    port_permit = 1'b1;
    @(posedge ck);

    test.subtitle = "(3.3) port 1 select. : output";
    expected_data = cont_comb_in.mo[3];
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(3.4) port 1 select. : input";
    cont_sel_so = 4'b1010;
    @(posedge ck);

    expected_data = '0;
    actual_data   = cont_comb_in.so[1];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(3.5) port 1 select. : port 2 nothing";
    expected_data = '0;
    actual_data   = cont_comb_in.so[2];
    test.assert_equals(expected_data, actual_data, "bin");

    test.subtitle = "(3.6) port 1 select. : port 3 nothing";
    expected_data = cont_sel_so;
    actual_data   = cont_comb_in.so[3];
    test.assert_equals(expected_data, actual_data, "bin");

    @(posedge ck);

    // permission を off にする。
    port_permit = 1'b0;

    @(posedge ck);

    // なにも選択されていない。
    test.subtitle = "(3.7) nothing is selected.";
    expected_data = '0;
    actual_data   = cont_sel_mo;
    test.assert_equals(expected_data, actual_data, "bin");

    cont_comb_in.mo[1].request  = 1'b0;
    cont_comb_in.mo[2].request  = 1'b0;
    cont_comb_in.mo[3].request  = 1'b0;
    @(posedge ck);






    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

