//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// spi_cont_switch の動作を確認する。
//   複数の request があるケースで、１回目のコマンドの実行後 dev_sts_read が
// 実行されることを確認する。 
//
//  (1) reset から 待ち状態に入る。
//  (2) 複数の request をセットする。
//  (3) セットした request の内、最も優先度の高いコマンドが実行される。 
//  (4) can_dev_sts_read が実行される。
//
module   test112_04_spi_cont_switch_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test112_04_spi_cont_switch_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

`include "can_msg.svh"
`include "spi_cont_comb.svh"


//-----------------------------------------------------------
// テスト用のパラメータ設定
localparam  PORT_NUM = 5;

//-----------------------------------------------------------
logic   ck, reset_n;
time    timeout;
logic   [127:0] expected, actual;

spi_cont_if     can_dev_sts_read_cmd();
spi_cont_if     device_config_cmd();
spi_cont_if     config_data_msg();
spi_cont_if     di_data_msg();
spi_cont_if     can_rcv_msg_read_cmd();

spi_cont_comb_if # (
    .PORT_NUM  (PORT_NUM)
) spi_cont_comb();


spi_cont_if     spi_cont_bus();

logic   interrupt;
logic   dev_sts_update;


//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


sim_spi_cont_sig_master     a_can_dev_sts_read_cmd (
    .cont_out       (can_dev_sts_read_cmd),
    .value_reset    (1'b0)
);

sim_spi_cont_sig_master     a_device_config_cmd (
    .cont_out       (device_config_cmd),
    .value_reset    (1'b0)
);

sim_spi_cont_sig_master     a_config_data_msg (
    .cont_out       (config_data_msg),
    .value_reset    (1'b0)
);

sim_spi_cont_sig_master     a_di_data_msg (
    .cont_out       (di_data_msg),
    .value_reset    (1'b0)
);

sim_spi_cont_sig_master     a_can_rcv_msg_read_cmd (
    .cont_out       (can_rcv_msg_read_cmd),
    .value_reset    (1'b0)
);


spi_cont_combiner   a_spi_cont_combiner ( 
    .port_1     (can_dev_sts_read_cmd),
    .port_2     (device_config_cmd),
    .port_3     (config_data_msg),
    .port_4     (di_data_msg),
    .port_5     (can_rcv_msg_read_cmd),
    .comb_out   (spi_cont_comb)
);


sim_spi_cont_sig_slave      a_spi_cont (
    .value_reset    (1'b0),
    .cont_in        (spi_cont_bus)
);



// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

spi_cont_switch  #(
    .PORT_NUM   (PORT_NUM)
) 
the_spi_cont_switch (
    .reset_n        (reset_n),
    .ck             (ck),
    .interrupt      (interrupt),
    .dev_sts_update (dev_sts_update),
    .cont_comb_in   (spi_cont_comb),
    .cont_out       (spi_cont_bus)
);


logic   int_or_request_wait_state;
logic   int_or_request_wait_state_n;
logic   request_check_state;

assign  int_or_request_wait_state =  (
    the_spi_cont_switch.curr_state == the_spi_cont_switch.s_int_or_request_wait
);
assign  int_or_request_wait_state_n = ~int_or_request_wait_state;

assign  request_check_state =  (
    the_spi_cont_switch.curr_state == the_spi_cont_switch.s_request_check
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    interrupt = 1'b0;
    actual = '0; expected = '0;
    timeout = 0;


    #100  reset_n = 1'b1;
    #500;

    @(posedge ck);

    //-----------------------------------------------------------------
    // (1) reset から 待ち状態に入る。

    // wait 状態が継続しているか確認する。
    test.subtitle = "(1.1) enter wait state.";
    test.assert_true(int_or_request_wait_state);

    test.subtitle = "(1.2) continue wait state.";
    timeout = SIM_COUNT_1MS;
    test.assert_not_occurr(int_or_request_wait_state_n, timeout);

    //-----------------------------------------------------------------
    //  (2) 複数の request をセットする。
    @(posedge ck);
    a_di_data_msg.request = 1'b1;
    @(posedge ck);
    a_config_data_msg.request = 1'b1;

    //-----------------------------------------------------------------
    //  (3) セットした request の内、最も優先度の高いコマンドが実行される。 
    // 制御権を許可する。(permit)
    test.subtitle = "(3.1) permit wait.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(a_config_data_msg.permit, timeout);


    #(SIM_COUNT_1US*100);


    // 制御権をリリースする。(request off)
    test.subtitle = "(3.2) permit off wait.";
    a_config_data_msg.request = 1'b0;
    timeout = SIM_COUNT_1US;
    test.assert_neg_edge(a_config_data_msg.permit, timeout);


    //-----------------------------------------------------------------
    //  (4) can_dev_sts_read が実行される。
    // can_dev_sts_read の start がかかる。
    test.subtitle = "(4.1) wait dev_sts_update.";
    timeout = SIM_COUNT_1US;
    test.assert_pos_edge(dev_sts_update, timeout);

    #(SIM_COUNT_1US);

    // can_dev_sts_read_cmd から request を出力する。 
    a_can_dev_sts_read_cmd.request = 1'b1;

    // 制御権を許可する。(permit)
    test.subtitle = "(4.2) permit wait.";
    timeout = SIM_CLK_COUNT * 10;
    test.assert_pos_edge(a_can_dev_sts_read_cmd.permit, timeout);

    can_dev_sts_read_cmd_signal_test("(4.3) signal connect test.");


    #(SIM_COUNT_1US*100);


    // 制御権をリリースする。(request off)
    test.subtitle = "(3.4) permit off wait.";
    a_can_dev_sts_read_cmd.request = 1'b0;
    test.assert_neg_edge(a_can_dev_sts_read_cmd.permit, timeout);









    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end


task device_config_cmd_signal_test(
        input   string  subtitle
    );

    //-------------------------------------------------
    // master to slave

    // byte_num
    test.subtitle = {subtitle, " : byte_num = 14"};
    expected = 14;
    a_device_config_cmd.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    test.subtitle = {subtitle, " : byte_num = 0"};
    expected = 0;
    a_device_config_cmd.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    // start
    test.subtitle = {subtitle, " : start = 1"};
    expected = 1'b1;
    a_device_config_cmd.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : start = 0"};
    expected = 1'b0;
    a_device_config_cmd.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    // mosi
    test.subtitle = {subtitle, " : mosi = 1"};
    expected = 1'b1;
    a_device_config_cmd.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : mosi = 0"};
    expected = 1'b0;
    a_device_config_cmd.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    //-------------------------------------------------
    // slave to master

    // done
    test.subtitle = {subtitle, " : done = 1"};
    expected = 1'b1;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_device_config_cmd.done;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : done = 0"};
    expected = 1'b0;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_device_config_cmd.done;
    test.assert_equals(expected, actual, "bin");

    // shift_carry
    test.subtitle = {subtitle, " : shift_carry = 1"};
    expected = 1'b1;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_device_config_cmd.shift_carry;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : shift_carry = 0"};
    expected = 1'b0;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_device_config_cmd.shift_carry;
    test.assert_equals(expected, actual, "bin");

    // miso
    test.subtitle = {subtitle, " : miso = 1"};
    expected = 1'b1;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_device_config_cmd.miso;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : miso = 0"};
    expected = 1'b0;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_device_config_cmd.miso;
    test.assert_equals(expected, actual, "bin");

endtask


task can_dev_sts_read_cmd_signal_test(
        input   string  subtitle
    );

    //-------------------------------------------------
    // master to slave

    // byte_num
    test.subtitle = {subtitle, " : byte_num = 14"};
    expected = 14;
    a_can_dev_sts_read_cmd.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    test.subtitle = {subtitle, " : byte_num = 0"};
    expected = 0;
    a_can_dev_sts_read_cmd.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    // start
    test.subtitle = {subtitle, " : start = 1"};
    expected = 1'b1;
    a_can_dev_sts_read_cmd.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : start = 0"};
    expected = 1'b0;
    a_can_dev_sts_read_cmd.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    // mosi
    test.subtitle = {subtitle, " : mosi = 1"};
    expected = 1'b1;
    a_can_dev_sts_read_cmd.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : mosi = 0"};
    expected = 1'b0;
    a_can_dev_sts_read_cmd.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    //-------------------------------------------------
    // slave to master

    // done
    test.subtitle = {subtitle, " : done = 1"};
    expected = 1'b1;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.done;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : done = 0"};
    expected = 1'b0;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.done;
    test.assert_equals(expected, actual, "bin");

    // shift_carry
    test.subtitle = {subtitle, " : shift_carry = 1"};
    expected = 1'b1;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.shift_carry;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : shift_carry = 0"};
    expected = 1'b0;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.shift_carry;
    test.assert_equals(expected, actual, "bin");

    // miso
    test.subtitle = {subtitle, " : miso = 1"};
    expected = 1'b1;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.miso;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : miso = 0"};
    expected = 1'b0;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_can_dev_sts_read_cmd.miso;
    test.assert_equals(expected, actual, "bin");

endtask


task config_data_msg_signal_test(
        input   string  subtitle
    );

    //-------------------------------------------------
    // master to slave

    // byte_num
    test.subtitle = {subtitle, " : byte_num = 14"};
    expected = 14;
    a_config_data_msg.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    test.subtitle = {subtitle, " : byte_num = 0"};
    expected = 0;
    a_config_data_msg.byte_num = expected;
    @(posedge ck);
    actual = a_spi_cont.byte_num;
    test.assert_equals(expected, actual, "int");

    // start
    test.subtitle = {subtitle, " : start = 1"};
    expected = 1'b1;
    a_config_data_msg.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : start = 0"};
    expected = 1'b0;
    a_config_data_msg.start = expected;
    @(posedge ck);
    actual = a_spi_cont.start;
    test.assert_equals(expected, actual, "bin");

    // mosi
    test.subtitle = {subtitle, " : mosi = 1"};
    expected = 1'b1;
    a_config_data_msg.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : mosi = 0"};
    expected = 1'b0;
    a_config_data_msg.mosi = expected;
    @(posedge ck);
    actual = a_spi_cont.mosi;
    test.assert_equals(expected, actual, "bin");

    //-------------------------------------------------
    // slave to master

    // done
    test.subtitle = {subtitle, " : done = 1"};
    expected = 1'b1;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_config_data_msg.done;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : done = 0"};
    expected = 1'b0;
    a_spi_cont.done = expected;
    @(posedge ck);
    actual = a_config_data_msg.done;
    test.assert_equals(expected, actual, "bin");

    // shift_carry
    test.subtitle = {subtitle, " : shift_carry = 1"};
    expected = 1'b1;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_config_data_msg.shift_carry;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : shift_carry = 0"};
    expected = 1'b0;
    a_spi_cont.shift_carry = expected;
    @(posedge ck);
    actual = a_config_data_msg.shift_carry;
    test.assert_equals(expected, actual, "bin");

    // miso
    test.subtitle = {subtitle, " : miso = 1"};
    expected = 1'b1;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_config_data_msg.miso;
    test.assert_equals(expected, actual, "bin");

    test.subtitle = {subtitle, " : miso = 0"};
    expected = 1'b0;
    a_spi_cont.miso = expected;
    @(posedge ck);
    actual = a_config_data_msg.miso;
    test.assert_equals(expected, actual, "bin");

endtask



endmodule

