//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) read  動作を確認する。
// (2) write 動作を確認する。
//
module   test113_01_fram_read_write_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test113_01_fram_read_write_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  INTERVAL = 40; // 1us 相当

// 使用するコマンドの最大サイズ (=n)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 

// fram_read_write 用の設定
`include "FM25L16B.svh"

localparam  BYTE_LEN = 8;
localparam  MSB = (BYTE_LEN*8) -1;

localparam  START_ADRS      = 16'h0123;
localparam  TEST_READ_DATA  = 64'h1122334455667788;
localparam  TEST_WRITE_DATA = 64'hABCDEF0123456789;



//-----------------------------------------------------------
logic   ck, reset_n;
logic   carry_1us, carry_gen_en;
time    timeout;
logic   expected_level, actual_level;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_cont_if spi_cont_bus();
spi_if      spi_bus();

logic   sck, cs;


logic   read;
logic   write;
logic   [15:0]  adrs;
logic   [MSB:0] din;
logic   [MSB:0] dout;
logic   done;

//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );

carry_gen #(
    .INTERVAL   (INTERVAL)
) 
a_carry_gen (
    .*,
    .en         (carry_gen_en),
    .carry      (carry_1us)
);



sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


sim_spi_cont #(
    .HALF_OF_SCK_PRI    (HALF_OF_SCK_PRI)
)
a_sim_spi_cont (
    .*,
    .carry      (carry_1us),
    .cont_in    (spi_cont_bus),
    .spi_port   (spi_bus)
); 

assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

fram_read_write # (
    .BYTE_LEN   (BYTE_LEN)
)
the_fram_read_write (
    .reset_n    (reset_n),
    .ck         (ck),
    .read       (read),
    .write      (write),
    .done       (done),
    .adrs       (adrs),
    .din        (din),
    .dout       (dout),
    .cont_out   (spi_cont_bus)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    read  = 1'b0;
    write = 1'b0;
    din = '0;
    adrs = '0;
    actual_level = 1'b0; expected_level = 1'b0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100;
    reset_n = 1'b1;
    carry_gen_en = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------
    // (1) read  動作を確認する。

    // start がかかる前は、 request は off 。
    test.subtitle = "(1.1) request is off before start.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");

    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = 3*8; // cmd(1) + adrs(2) byte;
    a_sim_spi_slave.res_bit_len = $bits(TEST_READ_DATA); // 8 byte;
    a_sim_spi_slave.response    = TEST_READ_DATA;
    adrs = START_ADRS;

    // start をかける。
    @(posedge ck) read = 1'b1;
    @(posedge ck) read = 1'b0;

    // request が出力されることを確認する。
    test.subtitle = "(1.2) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);

    // permit されるまで待機することを確認する。   
    // (sim_spi_cont() で 100us delay させている。)
    test.subtitle = "(1.3) permit wait.";
    timeout = SIM_COUNT_1US * 10;
    test.assert_not_occurr(spi_cont_bus.permit, timeout);
    timeout = SIM_COUNT_1US * 100;
    test.assert_pos_edge(spi_cont_bus.permit, timeout);

    // SPI 通信の完了を待つ。 (read command)
    test.subtitle = "(1.4) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    // command data と期待値を照合する。
    test.subtitle = "(1.5) FRAM read command test.";
    expected_data = {CMD_READ, START_ADRS};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");

    // 完了後 done が出力される。
    test.subtitle = "(1.6) done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(done, timeout);

    // read data と期待値を照合する。
    test.subtitle = "(1.7) FRAM read response test.";
    expected_data = TEST_READ_DATA;
    actual_data = dout;
    test.assert_equals(expected_data, actual_data, "hex");

    #(SIM_COUNT_1MS);

    //-----------------------------------------------------------------
    // (2) write 動作を確認する。

    // start がかかる前は、 request は off 。
    test.subtitle = "(2.1) request is off before start.";
    expected_level = 1'b0;
    actual_level   = spi_cont_bus.request;
    test.assert_equals(expected_level, actual_level, "bin");


    din = TEST_WRITE_DATA;

    // start をかける。
    @(posedge ck) write = 1'b1;
    @(posedge ck) write = 1'b0;

    // request が出力されることを確認する。
    test.subtitle = "(2.2) request wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.request, timeout);

    // permit されるまで待機することを確認する。   
    // (sim_spi_cont() で 100us delay させている。)
    test.subtitle = "(2.3) permit wait.";
    timeout = SIM_COUNT_1US * 10;
    test.assert_not_occurr(spi_cont_bus.permit, timeout);
    timeout = SIM_COUNT_1US * 100;
    test.assert_pos_edge(spi_cont_bus.permit, timeout);


    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = (1)*8; // cmd(1)
    a_sim_spi_slave.res_bit_len = '0;
    a_sim_spi_slave.response    = TEST_WRITE_DATA;

    // SPI 通信の完了を待つ。 (write enable command)
    test.subtitle = "(2.4) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    // command data と期待値を照合する。
    test.subtitle = "(2.5) FRAM write enable command test.";
    expected_data = {CMD_WREN};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = (1+2+BYTE_LEN)*8; // cmd(1) + adrs(2) + BYTE_LEN
    a_sim_spi_slave.res_bit_len = '0;
    a_sim_spi_slave.response    = TEST_WRITE_DATA;

    // SPI 通信の完了を待つ。 (write command)
    test.subtitle = "(2.6) spi cont done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(spi_cont_bus.done, timeout);

    // command data と期待値を照合する。
    test.subtitle = "(2.7) FRAM write command test.";
    expected_data = {CMD_WRITE, START_ADRS, TEST_WRITE_DATA};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    // 完了後 done が出力される。
    test.subtitle = "(2.8) done wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(done, timeout);




    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

