//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) read  command の動作を確認する。
// (2) write command の動作を確認する。
//
module   test114_01_config_data_manager_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test114_01_config_data_manager_tb\"";

//-----------------------------------------------------------
// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "dspl_value_tsk.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"
`include "shift_register_c.sv"


//-----------------------------------------------------------
// テスト用のパラメータ設定

localparam  INTERVAL = 40; // 1us 相当

// 使用するコマンドの最大サイズ (=n)より大きくする。
localparam  SREG_BYTE_LEN = 32;

localparam  HALF_OF_SCK_PRI = 2;
localparam  SCK_PRI = HALF_OF_SCK_PRI * 2;     // sck:10MHz 

// fram_read_write 用の設定
`include "FM25L16B.svh"

localparam  BYTE_LEN = 8;
localparam  MSB = (BYTE_LEN*8) -1;

localparam  START_ADRS      = 16'h0000;
localparam  TEST_READ_DATA  = 64'h1122334455667788;
localparam  TEST_WRITE_DATA = 64'hABCDEF0123456789;



//-----------------------------------------------------------
logic   ck, reset_n;
logic   carry_1us, carry_gen_en;
time    timeout;
logic   [(SREG_BYTE_LEN*8-1):0] expected_data;
logic   [(SREG_BYTE_LEN*8-1):0] actual_data;


spi_if      spi_bus();

logic   sck, cs;


logic   read_cmd;
logic   read_done;
logic   write_cmd;
logic   write_done;
logic   [MSB:0] din;
logic   [MSB:0] dout;

//-----------------------------------------------------------
//  テスト環境用のインスタンス
sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


carry_gen #(
    .INTERVAL   (INTERVAL)
) 
a_carry_gen (
    .*,
    .en         (carry_gen_en),
    .carry      (carry_1us)
);


sim_spi_slave #(
    .SREG_BYTE_LEN  (SREG_BYTE_LEN)
) 
a_sim_spi_slave (
    .spi_port  (spi_bus)
);


assign  sck = spi_bus.sck;
assign  cs  = spi_bus.cs;


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス 

config_data_manager # (
    .BYTE_LEN   (BYTE_LEN)
)
the_config_data_manager (
    .reset_n    (reset_n),
    .ck         (ck),
    .read_cmd   (read_cmd),
    .read_done  (read_done),
    .write_cmd  (write_cmd),
    .write_done (write_done),
    .din        (din),
    .dout       (dout),
    .spi_port   (spi_bus)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);
    test.title = title;

    reset_n = 1'b0;
    carry_gen_en = 1'b0;
    read_cmd  = 1'b0;
    write_cmd = 1'b0;
    din = '0;
    actual_data = '0; expected_data = '0;
    timeout = 0;


    #100;
    reset_n = 1'b1;
    carry_gen_en = 1'b1;
    #500;

    @(posedge ck);


    //-----------------------------------------------------------------
    // (1) read command 動作を確認する。

    // spi slave インスタンスの設定
    a_sim_spi_slave.cmd_bit_len = 3*8; // cmd(1) + adrs(2) byte;
    a_sim_spi_slave.res_bit_len = $bits(TEST_READ_DATA); // 8 byte;
    a_sim_spi_slave.response    = TEST_READ_DATA;

    // read command を実行する。
    @(posedge ck) read_cmd = 1'b1;
    @(posedge ck) read_cmd = 1'b0;

    // cs が on になるのを待つ。
    test.subtitle = "(1.1) cs on wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cs, timeout);

    // cs が off になるのを待つ。
    test.subtitle = "(1.2) cs off wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(cs, timeout);

    @(posedge ck);

    // command data と期待値を照合する。
    test.subtitle = "(1.3) FRAM read command test.";
    expected_data = {CMD_READ, START_ADRS};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");

    // read_done の立ち上がりを待つ。
    test.subtitle = "(1.4) read_done raise wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(read_done, timeout);

    test.subtitle = "(1.5) read_done fall wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(read_done, timeout);


    // dout の値と期待値を照合する。
    test.subtitle = "(1.6) dout test.";
    expected_data = TEST_READ_DATA;
    actual_data = dout;
    test.assert_equals(expected_data, actual_data, "hex");

    #(SIM_COUNT_1US * 10);


    //-----------------------------------------------------------------
    // (2) write command 動作を確認する。

    // spi slave インスタンスの設定(write enable)
    a_sim_spi_slave.cmd_bit_len = 1*8; // cmd(1) byte;
    a_sim_spi_slave.res_bit_len = 0;
    a_sim_spi_slave.response    = '0;

    // write command を実行する。
    din = TEST_WRITE_DATA;
    @(posedge ck) write_cmd = 1'b1;
    @(posedge ck) write_cmd = 1'b0;

    // cs が on になるのを待つ。(write enable)
    test.subtitle = "(2.1) cs on wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cs, timeout);

    // cs が off になるのを待つ。
    test.subtitle = "(2.2) cs off wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(cs, timeout);

    @(posedge ck);

    // command data と期待値を照合する。
    test.subtitle = "(2.3) FRAM write enable command test.";
    expected_data = {CMD_WREN};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    // spi slave インスタンスの設定(write)
    a_sim_spi_slave.cmd_bit_len = (1+2+BYTE_LEN)*8; // cmd(1) + adrs(2) + BYTE_LEN;
    a_sim_spi_slave.res_bit_len = '0;
    a_sim_spi_slave.response    = '0;

    // cs が on になるのを待つ。(write)
    test.subtitle = "(2.4) cs on wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cs, timeout);

    // cs が off になるのを待つ。
    test.subtitle = "(2.5) cs off wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(cs, timeout);

    @(posedge ck);

    // command data と期待値を照合する。
    test.subtitle = "(2.6) FRAM write command test.";
    expected_data = {CMD_WRITE, START_ADRS, TEST_WRITE_DATA};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    // spi slave インスタンスの設定(read)
    a_sim_spi_slave.cmd_bit_len = (1+2)*8; // cmd(1) + adrs(2);
    a_sim_spi_slave.res_bit_len = $bits(TEST_WRITE_DATA);
    a_sim_spi_slave.response    = TEST_WRITE_DATA;

    // cs が on になるのを待つ。(write 後の read)
    test.subtitle = "(2.7) cs on wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(cs, timeout);

    // dout は、まだ更新されていない。
    test.subtitle = "(2.8) dout has not been updated yet.";
    expected_data = dout;
    actual_data = TEST_READ_DATA;
    test.assert_equals(expected_data, actual_data, "hex");

    // cs が off になるのを待つ。
    test.subtitle = "(2.9) cs off wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(cs, timeout);

    @(posedge ck);

    // command data と期待値を照合する。
    test.subtitle = "(2.10) FRAM read command test.";
    expected_data = {CMD_READ, START_ADRS};
    actual_data = a_sim_spi_slave.rcv_command;
    test.assert_equals(expected_data, actual_data, "hex");


    // write_done の立ち上がりを待つ。
    test.subtitle = "(2.11) write_done raise wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_pos_edge(write_done, timeout);

    test.subtitle = "(2.12) write_done fall wait.";
    timeout = SIM_COUNT_1MS;
    test.assert_neg_edge(write_done, timeout);


    // dout の値と期待値を照合する。
    test.subtitle = "(2.13) dout test.";
    expected_data = TEST_WRITE_DATA;
    actual_data = dout;
    test.assert_equals(expected_data, actual_data, "hex");






    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);


end



endmodule

