//----------------------------------------------------------------------------
//   Copyright 2019  Simple Logic Systems Ltd.
//
//   Licensed under the Apache License, Version 2.0 (the "License");
//   you may not use this file except in compliance with the License.
//   You may obtain a copy of the License at
//
//       http://www.apache.org/licenses/LICENSE-2.0
//
//   Unless required by applicable law or agreed to in writing, software
//   distributed under the License is distributed on an "AS IS" BASIS,
//   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//   See the License for the specific language governing permissions and
//   limitations under the License.
//
//----------------------------------------------------------------------------
`default_nettype none

//
// (1) set trig を入力し、data_out にデータが出力されることを確認する。
//
module   test116_01_digital_output_tb;

`include "sim_timeunit.svh"
`include "sim_settings.vh"

`include "clock.svh"

localparam  title = "\"test116_01_digital_output_tb\"";

//-----------------------------------------------------------
logic   ck;
logic   reset_n;


localparam  SIG_BIT_WIDTH = 8;
localparam  DURATION_CNT = 3'd5;

localparam  MSB = SIG_BIT_WIDTH - 1;
logic   [MSB:0] data_in;
logic   [MSB:0] data_out;

logic   [MSB:0] expected_data;
logic   [MSB:0] actual_data;

logic   set;
logic   [MSB:0] test_data;
logic   [MSB:0] last_data_out;


// テスト用の class, task, function は、ここで展開する。 
`include "assertion_c.sv"
`include "timer_c.sv"
`include "test_timeout_tsk.sv"

//-----------------------------------------------------------
//  テスト環境用のインスタンス

sim_clock_gen #(
    .SIM_CLK_COUNT  (SIM_CLK_COUNT)
)     
a_sim_clock_gen( .ck );


// タイムアウトで無条件に中止する。
time test_timeout_cnt;
initial begin
    test_timeout_cnt = 100 * SIM_COUNT_1MS;
    test_timeout(test_timeout_cnt);
end


//-----------------------------------------------------------
//  テスト対象のインスタンス

digital_output #(
    .SIG_BIT_WIDTH  ($bits(data_in))
) 
the_digital_output (
    .reset_n   (reset_n),
    .ck        (ck),
    .set       (set),
    .data_in   (data_in),
    .data_out  (data_out)
);


//-----------------------------------------------------------
assertion_c test;
timer_c     tim1;


initial begin
    test = new();
    tim1 = new();

    $timeformat(-6, 3, " us", 12);

    test.title = title;
    reset_n = 1'b0;
    data_in = 8'h00;
    set = 1'b0;
    test_data = '0;
    last_data_out = '0;
    actual_data = '0; expected_data = '0;


    #100 reset_n = 1'b1;
    #500;

    @(posedge ck);


    //-------------------------------------------------------------
    // (1) set trig を入力し、data_out にデータが出力されることを確認する。

    test_data = 8'b10101010;
    data_in = test_data;

    // set trig を入力する前は、初期値のまま
    test.subtitle = "(1.1) data_out is zero before set trig.";
    expected_data = '0;
    actual_data = data_out;
    test.assert_equals(expected_data, actual_data, "bin");

    // set trig を入力する。
    @(posedge ck) set = 1'b1;
    @(posedge ck) set = 1'b0;
    @(posedge ck);

    // 出力と期待値を照合する。
    test.subtitle = "(1.2) data_out test.";
    expected_data = test_data;
    actual_data = data_out;
    test.assert_equals(expected_data, actual_data, "bin");
    last_data_out = data_out;

    #(SIM_COUNT_1US);

    test_data = 8'b01010101;
    data_in = test_data;

    #(SIM_COUNT_1US);

    // set trig を入力する前は、前回値のまま
    test.subtitle = "(1.3) data_out has not been changed before set trig.";
    expected_data = last_data_out;
    actual_data = data_out;
    test.assert_equals(expected_data, actual_data, "bin");

    #(SIM_COUNT_1US);

    // set trig を入力する。
    @(posedge ck) set = 1'b1;
    @(posedge ck) set = 1'b0;
    @(posedge ck);

    // 出力と期待値を照合する。
    test.subtitle = "(1.4) data_out test.";
    expected_data = test_data;
    actual_data = data_out;
    test.assert_equals(expected_data, actual_data, "bin");






    $display("\n");
    $display("%s passed.", title);
    $display("\n");

    $finish(0);

end




endmodule

