"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""

from enum import IntEnum
from simgics.s1.can.errors import UnknownMessageError


class CanMessageFrame():
    """CAN Message Frame を取り扱うクラス.

    外部のCAN通信ライブラリで用意されているクラスは直接使用せず、
    必要最低限の要素に絞ってメンバーとしている。
    """

    def __init__(self):
        self.arbitration_id = 0
        self.is_error_frame = False
        self.is_remote_frame = False
        self.is_ext_id = True
        self.dlc = 0
        self.data = bytearray()


    def __eq__(self, other):
        ret = (
            self.arbitration_id == other.arbitration_id and
            self.is_error_frame == other.is_error_frame and
            self.is_remote_frame == other.is_remote_frame and
            self.is_ext_id == other.is_ext_id and
            self.dlc == other.dlc and
            self.data == other.data
            )

        return ret


    def __ne__(self, other):
        return not( self.__eq__(other) )



class AppMsgId(IntEnum):
    STATUS_REQUEST = 0x01
    CONFIG_DATA_REQUEST = 0x02
    CONFIG_DATA_WRITE = 0x03
    DO_DATA_SET = 0x04
    CONFIG_DATA = 0x81
    DI_DATA_STATUS = 0x82
    DO_DATA_STATUS = 0x83



class ApplicationMessage():
    """各 Application Message の基底クラス"""

    def __init__(self):
        self.__src_stn_id = 0
        self.__dest_stn_id = 0
        self.__msg_id = 0
        self.__dlc = 0
        self.__data = bytearray()


    def __str__(self):
        return self._message_display()


    def _message_display(self):
        ret = "{:<22}, ".format(self.__class__.__name__) + \
            "src_id:0x{:02x}, ".format(self.src_stn_id) + \
            "dest_id:0x{:02x}, ".format(self.dest_stn_id) + \
            "msg_id:0x{:02x}, ".format(self.msg_id) +\
            "dlc:{:2d}, ".format(self.dlc) + \
            "data:0x" + self.data.hex()

        return ret


    @property
    def src_stn_id(self):
        return self.__src_stn_id

    @src_stn_id.setter
    def src_stn_id(self, stn_id):
        if not(0<=stn_id<=255) :
            msg = 'Illegal src_stn_id: 0x{:02X}'.format(stn_id)
            raise ValueError(msg)

        self.__src_stn_id = stn_id


    @property
    def dest_stn_id(self):
        return self.__dest_stn_id

    @dest_stn_id.setter
    def dest_stn_id(self, stn_id):
        if not(0<=stn_id<=255) :
            msg = 'Illegal dest_stn_id: 0x{:02X}'.format(stn_id)
            raise ValueError(msg)

        self.__dest_stn_id = stn_id


    @property
    def msg_id(self):
        return self.__msg_id

    @msg_id.setter
    def msg_id(self, msg_id):
        if not(0<=msg_id<=255) :
            msg = 'Illegal msg_id: 0x{:02X}'.format(msg_id)
            raise ValueError(msg)

        self.__msg_id = msg_id


    @property
    def dlc(self):
        return self.__dlc

    @dlc.setter
    def dlc(self, dlc):
        if not(0<=dlc<=15) :
            msg = 'Illegal dlc: {}'.format(dlc)
            raise ValueError(msg)

        self.__dlc = dlc


    @property
    def data(self):
        return self.__data

    @data.setter
    def data(self, data):
        data_len = len(data)
        if not(0<=data_len<=8) :
            msg = 'Illegal data length: {}'.format(data_len)
            raise ValueError(msg)

        self.__data = data.copy()


    def get_can_msg_frame(self):
        can_msg_frm = CanMessageFrame()

        can_msg_frm.arbitration_id = (
            (self.src_stn_id << 20) +
            (self.msg_id << 8) +
            (self.dest_stn_id)
            )
        can_msg_frm.dlc = self.dlc
        can_msg_frm.data = self.data.copy()

        return can_msg_frm



class StatusRequestMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dest_stn_id = 0):
        super().__init__()
        self.msg_id = AppMsgId.STATUS_REQUEST
        self.src_stn_id = src_stn_id
        self.dest_stn_id = dest_stn_id


class ConfigDataRequestMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dest_stn_id):
        super().__init__()
        self.msg_id = AppMsgId.CONFIG_DATA_REQUEST
        self.src_stn_id = src_stn_id
        self.dest_stn_id = dest_stn_id


class ConfigDataWriteMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dest_stn_id, dlc, data: bytearray):
        super().__init__()
        self.msg_id = AppMsgId.CONFIG_DATA_WRITE
        self.src_stn_id = src_stn_id
        self.dest_stn_id = dest_stn_id
        self.data = data
        self.dlc = dlc


class DoDataSetMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dest_stn_id, dlc, data: bytearray):
        super().__init__()
        self.msg_id = AppMsgId.DO_DATA_SET
        self.src_stn_id = src_stn_id
        self.dest_stn_id = dest_stn_id
        self.data = data
        self.dlc = dlc


class ConfigDataMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dlc, data):
        super().__init__()
        self.msg_id = AppMsgId.CONFIG_DATA
        self.src_stn_id = src_stn_id
        self.dlc = dlc
        self.data = data


class DiDataStatusMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dlc, data):
        super().__init__()
        self.msg_id = AppMsgId.DI_DATA_STATUS
        self.src_stn_id = src_stn_id
        self.dlc = dlc
        self.data = data


class DoDataStatusMsg(ApplicationMessage):
    def __init__(self, src_stn_id, dlc, data):
        super().__init__()
        self.msg_id = AppMsgId.DO_DATA_STATUS
        self.src_stn_id = src_stn_id
        self.dlc = dlc
        self.data = data



class AppMsgFactory():
    SRC_STN_ID_BIT_POS = 20
    MSG_ID_BIT_POS = 8
    DEST_STN_ID_BIT_POS = 0


    @classmethod
    def create(cls, msg: CanMessageFrame):
        """Application Message オブジェクトを生成する.

        msg_id を調べて、それに応じて生成処理を切り替える。

         Args:
            msg: CAN message frame.

        Raises:
            UnknownMessageError: サポート外の message を受信した.

       """
        src_stn_id = (msg.arbitration_id >> cls.SRC_STN_ID_BIT_POS) & 0xff
        msg_id = (msg.arbitration_id >> cls.MSG_ID_BIT_POS) & 0xff
        dest_stn_id = (msg.arbitration_id >> cls.DEST_STN_ID_BIT_POS) & 0xff
        dlc = msg.dlc
        data = msg.data


        if msg_id == AppMsgId.STATUS_REQUEST :
            app_msg = StatusRequestMsg(src_stn_id, dest_stn_id)
        elif msg_id == AppMsgId.CONFIG_DATA_REQUEST :
            app_msg = ConfigDataRequestMsg(src_stn_id, dest_stn_id)
        elif msg_id == AppMsgId.CONFIG_DATA_WRITE :
            app_msg = ConfigDataWriteMsg(src_stn_id, dest_stn_id, dlc, data)
        elif msg_id == AppMsgId.DO_DATA_SET :
            app_msg = DoDataSetMsg(src_stn_id, dest_stn_id, dlc, data)
        elif msg_id == AppMsgId.CONFIG_DATA :
            app_msg = ConfigDataMsg(src_stn_id, dlc, data)
        elif msg_id == AppMsgId.DI_DATA_STATUS :
            app_msg = DiDataStatusMsg(src_stn_id, dlc, data)
        elif msg_id == AppMsgId.DO_DATA_STATUS :
            app_msg = DoDataStatusMsg(src_stn_id, dlc, data)
        else:
            msg = "unknown msg_id: 0x{:02X}".format(msg_id)
            raise UnknownMessageError(msg)


        return app_msg



def data_str_to_bytearray(data_str, length):
    """数値文字列 を length バイトの bytearray へ変換する."""
    val = int(data_str, 0)
    byte_arr = bytearray(val.to_bytes(length, byteorder='big'))

    return byte_arr



