"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""
import can

from simgics.s1.can.application_message import CanMessageFrame
from simgics.s1.can.errors import CanCommunicationError, CanReceiveTimeout


class CanPort():
    """CAN 通信を行うクラス.

    外部ライブラリを本クラスにカプセル化して、外部ライブラリ
    を変更した場合に影響を少なくすることを目的としている。
    """

    def __init__(self, params = {"channel" :'test', "bitrate" : 0}):
        if params['channel'] == 'test':
            # PC 上での開発を進める間は virtual bus でテストを行う。
            self._bus = can.interface.Bus('test', bustype='virtual')
        else:
            self._bus = can.interface.Bus(
                bustype = 'socketcan',
                channel = params['channel'],
                bitrate = params['bitrate']
            )

    def _to_can_msg_frm(self, can_msg: can.Message):
        """can.Message を CanMessageFrame に変換する."""
        msg_frm = CanMessageFrame()

        msg_frm.arbitration_id = can_msg.arbitration_id
        msg_frm.is_ext_id = can_msg.is_extended_id
        msg_frm.is_remote_frame = can_msg.is_remote_frame
        msg_frm.is_error_frame = can_msg.is_error_frame
        msg_frm.dlc = can_msg.dlc
        msg_frm.data = can_msg.data.copy()

        return msg_frm


    def _to_can_message(self, msg_frm: CanMessageFrame):
        """CanMessageFrame を can.Message に変換する."""
        can_msg = can.Message()

        can_msg.arbitration_id = msg_frm.arbitration_id
        can_msg.dlc = msg_frm.dlc
        can_msg.data = msg_frm.data
        can_msg.is_extended_id = msg_frm.is_ext_id
        can_msg.is_remote_frame = msg_frm.is_remote_frame
        can_msg.is_error_frame = msg_frm.is_error_frame

        return can_msg


    def send(self, msg_frm: CanMessageFrame):
        """CAN message を送信する.

        Args:
            msg_frm: 送信する message frame.

        Raises:
            CanCommunicationError: CAN通信のエラー.

        """
        can_msg = self._to_can_message(msg_frm)

        try:
            self._bus.send(can_msg)
        except can.CanError as ex:
            print(ex)
            raise CanCommunicationError("CAN send error.")


    def recv(self, timeout_s : float):
        """CAN message を受信する.

        Args:
            timeout_s(float): 受信タイムアウト値(sec).

        Returns:
            CanMessageFrame: 受信した message.

        Raises:
            CanCommunicationError: CAN通信のエラー.
            CanReceiveTimeout    : 受信タイムアウト.

        """
        try:
            can_msg = self._bus.recv(timeout_s)
        except can.CanError as ex:
            print(ex)
            raise CanCommunicationError("CAN recv error.")

        if can_msg is None:
            raise CanReceiveTimeout("CAN recv timeout.")

        msg_frm = self._to_can_msg_frm(can_msg)

        return msg_frm



