"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""
import traceback
import argparse

from simgics.s1.can.can_port import CanPort
from simgics.s1.can.application_message import ConfigDataWriteMsg
from simgics.s1.can.application_message import data_str_to_bytearray


class ConfigWrite():
    _CHANNEL = 'can0'
    _BIT_RATE = 1000000 # 1Mbps
    DATA_LEN = 8 # config data のバイト数は 8

    def __init__(self, src_stn_id, dest_stn_id, data, is_test=False):
        self._src_stn_id = src_stn_id
        self._dest_stn_id = dest_stn_id
        self._data = data

        if is_test :
            self._port = CanPort()
        else:
            params = {"channel" :self._CHANNEL, "bitrate" : self._BIT_RATE}
            self._port = CanPort(params)


    def _str_to_bytearray(self, data_str: str):
        """data_str をsize 8 の bytearray へ変換する.

                  MSB       LSB
        bytearray [0] ..... [7]
        """
        val = int(data_str, 0)
        byte_arr = bytearray(val.to_bytes(self.DATA_LEN, byteorder='big'))

        return byte_arr


    def _send_config_data_write_msg(self):
        app_msg = ConfigDataWriteMsg(self._src_stn_id, self._dest_stn_id, 8, self._data)
        msg_frm = app_msg.get_can_msg_frame()
        self._port.send(msg_frm)


    def execute(self):
        """Config Write を実行する.

        (1) config data write message を送出する。
        """
        self._send_config_data_write_msg()


def main():
    parser = argparse.ArgumentParser(
        description="config data write message を送出する."
        )
    parser.add_argument("dest_stn_id", help="宛先 station id.")
    parser.add_argument("data", help="config data. ex:0x1122334455667788")
    parser.add_argument("--src_stn_id",  help="送信元 station id.(省略時は 0x01)")
    args = parser.parse_args()

    src_stn_id = 0x01 if args.src_stn_id is None else int(args.src_stn_id, 0)
    dest_stn_id = 0x00 if args.dest_stn_id is None else int(args.dest_stn_id, 0)
    data = data_str_to_bytearray(args.data, ConfigWrite.DATA_LEN)

    config_write = ConfigWrite(src_stn_id, dest_stn_id, data)

    try:
        config_write.execute()

    except Exception as ex:
        print(ex.message)
        traceback.print_exc()



if __name__ == '__main__':
    main()
