"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""
import traceback
import argparse
from threading import Thread
from queue import Queue, Empty
import datetime

from simgics.s1.can.can_port import CanPort
from simgics.s1.can.application_message import StatusRequestMsg
from simgics.s1.can.application_message import AppMsgFactory
from simgics.s1.can.errors import CanCommunicationError, CanReceiveTimeout, UnknownMessageError


class MessageMonitor():
    _CHANNEL = 'can0'
    _BIT_RATE = 1000000 # 1Mbps

    def __init__(self, src_stn_id, is_test=False):
        self._src_stn_id = src_stn_id
        self._dest_stn_id = 0x00

        if is_test :
            self._port = CanPort()
        else:
            params = {"channel" :self._CHANNEL, "bitrate" : self._BIT_RATE}
            self._port = CanPort(params)

        self._msg_frm_buff = Queue(100)

        self._run = False
        self._recv_thread = Thread(target=self._msg_frm_recv)


    def _recv_thread_start(self):
        self._run = True
        self._recv_thread.start()


    def recv_thread_stop(self):
        self._run = False
        self._recv_thread.join(3) # 3 sec


    def _send_request(self):
        app_msg = StatusRequestMsg(self._src_stn_id, self._dest_stn_id)
        msg_frm = app_msg.get_can_msg_frame()
        self._port.send(msg_frm)


    def _msg_frm_recv(self):
        """受信スレッドとして動作させるメソッド.

        run の間、受信処理を繰り返す。
        """
        timeout_s = 2.0

        while self._run:

            try:
                msg_frm = self._port.recv(timeout_s)

            except CanReceiveTimeout as ex:
                continue

            except CanCommunicationError as ex:
                print(ex)
                continue

            if msg_frm is None:
                continue

            try:
                self._msg_frm_buff.put_nowait(msg_frm)

            except Exception as ex:
                print(ex)
                continue


    def execute(self):
        """Message Monitor を実行する.

        (1) 受信スレッドを開始する。
        (2) status request message を送出する。 bus に接続されている
          station から status message が送出される。
        (3) バッファから CAN message frame を取り出して、表示する操作を
          繰り返す。
            バッファが空になったら、Empty例外が発生するのがループ
          を繰り返す。

        （プログラムは ctrl + c で終了させる。）
        """
        self._recv_thread_start()
        self._send_request()

        timeout_s = 2.0
        while True:
            try:
                msg_frm = self._msg_frm_buff.get(timeout=timeout_s)

            except Empty:
                # buffer が空になってもループを繰り返す。
                continue

            try:
                now = datetime.datetime.now()
                app_msg = AppMsgFactory.create(msg_frm)
                print(now, app_msg)

            except UnknownMessageError as ex:
                print(ex)


def main():
    parser = argparse.ArgumentParser(
        description="各 station が送出する message をモニターする."
        )
    parser.add_argument("--src_stn_id",  help="送信元 station id.(省略時は 0x01)")
    args = parser.parse_args()

    src_stn_id = 0x01 if args.src_stn_id is None else int(args.src_stn_id, 0)

    msg_mon = MessageMonitor(src_stn_id)

    try:
        msg_mon.execute()

    except KeyboardInterrupt:
        print("プログラムを終了します。")
        msg_mon.recv_thread_stop()

    except Exception as ex:
        print(ex.message)
        traceback.print_exc()



if __name__ == '__main__':
    main()
