"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""

import unittest

from simgics.s1.can.application_message import AppMsgFactory
from simgics.s1.can.application_message import (
    CanMessageFrame, StatusRequestMsg, ConfigDataRequestMsg, ConfigDataWriteMsg,
    DoDataSetMsg, ConfigDataMsg, DiDataStatusMsg, DoDataStatusMsg
    )
from simgics.s1.can.errors import IllegalMessageError, UnknownMessageError


class AppMsgFactoryTestCase(unittest.TestCase):

    def test_create_status_request_msg(self):
        """StatusRequestMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x01
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500109 # 計算式を使わずマニュアルで値をセットする。

        dlc = 0
        data = bytearray()

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, StatusRequestMsg))
        self.assertTrue(expected == actual)


    def test_create_config_data_request_msg(self):
        """ConfigDataRequestMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x02
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500209 # 計算式を使わずマニュアルで値をセットする。

        dlc = 0
        data = bytearray()

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, ConfigDataRequestMsg))
        self.assertTrue(expected == actual)


    def test_create_config_data_write_msg(self):
        """ConfigDataWriteMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x03
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500309 # 計算式を使わずマニュアルで値をセットする。

        dlc = 2
        data = bytearray((0xab, 0xcd))

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, ConfigDataWriteMsg))
        self.assertTrue(expected == actual)


    def test_create_do_data_set_msg(self):
        """DoDataSetMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x04
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500409 # 計算式を使わずマニュアルで値をセットする。

        dlc = 2
        data = bytearray((0xab, 0xcd))

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, DoDataSetMsg))
        self.assertTrue(expected == actual)


    def test_create_config_data_msg(self):
        """ConfigDataMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x81
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508100 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, ConfigDataMsg))
        self.assertTrue(expected == actual)


    def test_create_di_data_status_msg(self):
        """DiDataStatusMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x82
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508200 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, DiDataStatusMsg))
        self.assertTrue(expected == actual)


    def test_create_do_data_status_msg(self):
        """DoDataStatusMsg の生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x83
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508300 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))

        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        app_msg = AppMsgFactory.create(expected)
        actual = app_msg.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(isinstance(app_msg, DoDataStatusMsg))
        self.assertTrue(expected == actual)


    def test_create_when_unknown_msg_id(self):
        """未定義の msg_id を入力した場合の動作を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0xAB <-- Unknown Msg ID
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x050AB00 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))

        unknown_msg = CanMessageFrame()
        unknown_msg.arbitration_id = arbitration_id
        unknown_msg.dlc = dlc
        unknown_msg.data = data

        #---  Verify  ------------------
        with self.assertRaises(UnknownMessageError):
            AppMsgFactory.create(unknown_msg)




