"""
  Copyright 2019  Simple Logic Systems Ltd.

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

"""

import unittest

from simgics.s1.can.application_message import (
    CanMessageFrame, ApplicationMessage,
    StatusRequestMsg, ConfigDataRequestMsg, ConfigDataWriteMsg, DoDataSetMsg,
    ConfigDataMsg, DiDataStatusMsg, DoDataStatusMsg
    )


class ApplicationMessageTestCase(unittest.TestCase):

    def test_app_msg_error_when_illegal_src_stn_id_set(self):
        #---  Setup  -------------------
        target = ApplicationMessage()

        #---  Verify  ------------------
        with self.assertRaises(ValueError):
            target.src_stn_id = -1

        target.src_stn_id = 0
        target.src_stn_id = 255

        with self.assertRaises(ValueError):
            target.src_stn_id = 256


    def test_app_msg_error_when_illegal_dest_stn_id_set(self):
        #---  Setup  -------------------
        target = ApplicationMessage()

        #---  Verify  ------------------
        with self.assertRaises(ValueError):
            target.dest_stn_id = -1

        target.dest_stn_id = 0
        target.dest_stn_id = 255

        with self.assertRaises(ValueError):
            target.dest_stn_id = 256


    def test_app_msg_error_when_illegal_msg_id_set(self):
        #---  Setup  -------------------
        target = ApplicationMessage()

        #---  Verify  ------------------
        with self.assertRaises(ValueError):
            target.msg_id = -1

        target.msg_id = 0
        target.msg_id = 255

        with self.assertRaises(ValueError):
            target.msg_id = 256


    def test_app_msg_error_when_illegal_dlc_set(self):
        #---  Setup  -------------------
        target = ApplicationMessage()

        #---  Verify  ------------------
        with self.assertRaises(ValueError):
            target.dlc = -1

        target.dlc = 0
        target.dlc = 15

        with self.assertRaises(ValueError):
            target.dlc = 16


    def test_app_msg_error_when_illegal_data_set(self):
        #---  Setup  -------------------
        target = ApplicationMessage()

        #---  Verify  ------------------
        target.data = bytearray()
        target.data = bytearray((1,))
        target.data = bytearray((1,2,3,4,5,6,7,8))

        with self.assertRaises(ValueError):
            target.data = bytearray((1,2,3,4,5,6,7,8,9))





    def test_get_can_message_1(self):
        """ApplicationMessage の get_can_msg_frame をテストする."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 2
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x03
        # arbitration_id[8:0]   : dest_stn_id = 4
        arbitration_id = 0x0200304 # 計算式を使わずマニュアルで値をセットする。

        dlc = 8
        data = bytearray((1,2,3,4,5,6,7,8))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        target = ApplicationMessage()
        target.src_stn_id = 2
        target.dest_stn_id = 4
        target.msg_id = 3
        target.dlc = 8
        target.data = bytearray((1,2,3,4,5,6,7,8))

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_get_can_message_2(self):
        """ApplicationMessage の get_can_msg_frame をテストする."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 2
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x81
        # arbitration_id[8:0]   : dest_stn_id = 4
        arbitration_id = 0x0208104 # 計算式を使わずマニュアルで値をセットする。

        dlc = 5
        data = bytearray((1,2,3,4,5))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        target = ApplicationMessage()
        target.src_stn_id = 2
        target.dest_stn_id = 4
        target.msg_id = 0x81
        target.dlc = 5
        target.data = bytearray((1,2,3,4,5))

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_status_request_msg(self):
        """StatusRequestMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x01
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500109 # 計算式を使わずマニュアルで値をセットする。

        dlc = 0
        data = bytearray()


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        dest_stn_id = 9
        target = StatusRequestMsg(src_stn_id, dest_stn_id)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_config_data_request_msg(self):
        """ConfigDataRequestMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x02
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500209 # 計算式を使わずマニュアルで値をセットする。

        dlc = 0
        data = bytearray()


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        dest_stn_id = 9
        target = ConfigDataRequestMsg(src_stn_id, dest_stn_id)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_config_data_write_msg(self):
        """ConfigDataWriteMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x03
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500309 # 計算式を使わずマニュアルで値をセットする。

        dlc = 2
        data = bytearray((0xab, 0xcd))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        dest_stn_id = 9
        dlc = 2
        data = bytearray((0xab, 0xcd))
        target = ConfigDataWriteMsg(src_stn_id, dest_stn_id, dlc, data)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_do_data_set_msg(self):
        """DoDataSetMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x04
        # arbitration_id[8:0]   : dest_stn_id = 9
        arbitration_id = 0x0500409 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        dest_stn_id = 9
        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))
        target = DoDataSetMsg(src_stn_id, dest_stn_id, dlc, data)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_config_data_msg(self):
        """ConfigDataMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x81
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508100 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        data = bytearray((0xab, 0xcd, 0xef))
        dlc = 3
        target = ConfigDataMsg(src_stn_id, dlc, data)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_di_data_status_msg(self):
        """DiDataStatusMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x82
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508200 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        data = bytearray((0xab, 0xcd, 0xef))
        dlc = 3
        target = DiDataStatusMsg(src_stn_id, dlc, data)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_do_data_status_msg(self):
        """DoDataStatusMsg の CanMessageFrame 生成を確認する."""

        #---  Setup  -------------------
        # arbitration_id[28]    : 0b0
        # arbitration_id[27:20] : src_stn_id = 5
        # arbitration_id[19:16] : 0b0000
        # arbitration_id[15:8]  : msg_id = 0x83
        # arbitration_id[8:0]   : 0x00
        arbitration_id = 0x0508300 # 計算式を使わずマニュアルで値をセットする。

        dlc = 3
        data = bytearray((0xab, 0xcd, 0xef))


        expected = CanMessageFrame()
        expected.arbitration_id = arbitration_id
        expected.dlc = dlc
        expected.data = data

        #---  Exercise  ----------------
        src_stn_id = 5
        data = bytearray((0xab, 0xcd, 0xef))
        dlc = 3
        target = DoDataStatusMsg(src_stn_id, dlc, data)

        actual = target.get_can_msg_frame()

        #---  Verify  ------------------
        self.assertTrue(expected == actual)


    def test_string_method_when_do_data_set_msg(self):
        """DoDataSetMsg の __str__ 出力文字列を確認する."""

        #---  Setup  -------------------
        src_stn_id = 5
        dest_stn_id = 9
        dlc = 3
        data = bytearray((0x00, 0xcd, 0xef))
        target = DoDataSetMsg(src_stn_id, dest_stn_id, dlc, data)

        expected = "DoDataSetMsg          , src_id:0x05, dest_id:0x09, " + \
            "msg_id:0x04, dlc: 3, data:0x00cdef"

        #---  Exercise  ----------------
        actual = str(target)

        #---  Verify  ------------------
        self.assertEqual(expected, actual)
